#include <stdio.h>

#include "../core/utils/utils.h"
#include "../core/debug/evemon.h"
#include "../core/debug/exception.h"
#include "../core/threads/event.h"
#include "../core/net/net.h"
#include "../core/net/packet.h"
#include "../core/net/tcp/tcp.h"
#include "../core/storage/base.h"

/* test class for packet */
class packet_test : public net::packet
{
public:
	packet_test(void) : packet(0xfeedbeef) {}

	virtual void on_process(const unsigned char* bytes, const size_t size)
	{
		this->m_buf = storage::buffer(bytes, size);
	}

	storage::buffer m_buf;
};

/* welcome message */
class msg_welcome : public net::tcp::msg
{
public:
	IMPLEMENT_STORAGE;
};

DECLARE_EMPTY_STORAGE(msg_welcome, net_tcp_msg_t);

/* ack message */
class msg_ack : public net::tcp::msg
{
public:
	IMPLEMENT_STORAGE;
};

DECLARE_EMPTY_STORAGE(msg_ack, net_tcp_msg_t);

/* simple client class */
class simple_client : public net::tcp::client<0xfeedbeef>
{
public:
	simple_client(const char* address) : client<0xfeedbeef>(address, 27015)
	{
		this->m_event.reset();

		REGISTER_MESSAGE_OBJECT(simple_client, msg_welcome);
	}

	bool wait(const double timeout) const
	{
		return this->m_event.wait(timeout);
	}

	bool do_until(std::function<void(void)> func, const double timeout) const
	{
		return this->m_event.do_until(func, timeout);
	}

	bool is_pending(void) const
	{
		return this->m_event.is_pending();
	}

	void reset(void)
	{
		this->m_event.reset();
	}

	HANDLE_MESSAGE(msg_welcome, msg)
	{
		this->m_event.trigger();

		// send ack message
		send(msg_ack());
	}

private:
	thread::event m_event;
};

/* simple server class */
class simple_server : public net::tcp::server<0xfeedbeef>
{
private:
	class client : public server::client
	{
	public:
		client(simple_server& parent, SOCKET socket) : server::client(parent, socket), m_parent(parent)
		{
			REGISTER_MESSAGE_OBJECT(client, msg_ack);
		}

		HANDLE_MESSAGE(msg_ack, msg)
		{
			this->m_parent.m_event.trigger();
		}

	private:
		simple_server& m_parent;
	};

public:
	simple_server(const char* address) : server<0xfeedbeef>(address, 27015)
	{
		this->m_event.reset();
	}

	bool wait(const double timeout) const
	{
		return this->m_event.wait(timeout);
	}

	bool is_pending(void) const
	{
		return this->m_event.is_pending();
	}

	void reset(void)
	{
		this->m_event.reset();
	}

	bool do_until(std::function<void(void)> func, const double timeout) const
	{
		return this->m_event.do_until(func, timeout);
	}

private:
	virtual std::shared_ptr<server::client> create(SOCKET socket) override
	{
		return std::make_shared<client>(*this, socket);
	}

	virtual void on_connect(std::shared_ptr<server::client> p) override
	{
		server::on_connect(p);

		this->m_event.trigger();
	}

	thread::event m_event;
};

#define SWITCHING

// main program
int main(int argc, char** argv)
{
	try
	{
		// initialize net interface
		net::init();

		/*
		 *	example 1:
		 *	test packet class
		 */
		printf("example 1:\r\n");
		printf("----------\r\n");
		printf("\r\n");

		{
			const char* text = "Hello World!";

			storage::buffer buf;

			// pack
			{
				packet_test tmp;

				printf("packing message...\r\n");

				buf = tmp.pack((const unsigned char*)text, strlen(text) + 1);
			}

			// unpack
			{
				packet_test tmp;

				tmp.unpack(buf.data(), buf.size());

				if (_strcmpi((const char*)tmp.m_buf.data(), text) != 0)
				{
					_error("Cannot unpack message!");

					return 1;
				}

				printf("message unpacked \"%s\"\r\n", (const char*)tmp.m_buf.data());
			}
		}

		printf("\r\n");

		/*
		 *	example 2:
		 *	simple tcp client
		 */
		printf("example 2:\r\n");
		printf("----------\r\n");
		printf("\r\n");

		{
			simple_server server("localhost");

			// start server
			printf("starting server...\r\n");

#ifndef SWITCHING
			server.start();
#endif

			{
				simple_client client("localhost");

				// start client
				printf("starting client...\r\n");

				{
					bool success = false;

#ifdef SWITCHING

					success = server.do_until([&](void)
						{
							if (!server.sync_run())
								std::this_thread::sleep_for(std::chrono::milliseconds(10));
						}, 1.0);

#else
					client.start();

					success = server.wait(1.0);
#endif

					// check that we have received connection
					if (!success)
					{
						_error("No connection received!");

						return 1;
					}
				}

				printf("client has connected\r\n");

				// server send welcome message
				printf("sending welcome message...\r\n");

				server.reset();

				server.send_all(msg_welcome());

				{
					bool success = false;

#ifdef SWITCHING

					success = client.do_until([&](void)
						{
							if (!client.sync_run())
								std::this_thread::sleep_for(std::chrono::milliseconds(10));
						}, 1.0);

#else
					client.start();

					success = client.wait(1.0);
#endif

					// check that we have received connection
					if (!success)
					{
						_error("Message not received!");

						return 1;
					}
			}

				printf("welcome message received\r\n");

				{
					bool success = false;

#ifdef SWITCHING

					success = server.do_until([&](void)
						{
							if (!server.sync_run())
								std::this_thread::sleep_for(std::chrono::milliseconds(10));
						}, 1.0);

#else
					client.start();

					success = server.wait(1.0);
#endif

					// check that we have received connection
					if (!success)
					{
						_error("No ack received!");

						return 1;
					}
			}

				printf("ack message received\r\n");

#ifndef SWITCHING
				// stop client
				client.stop();
#endif
			}

#ifndef SWITCHING
			// stop server gracefuly
			server.stop();
#endif
		}

		// close net interface
		net::close();

		// return successfuly
		return 0;
	}
	catch (exception::base& e)
	{
		printf("error: %s\r\n", e.to_string().c_str());

		_critical("%s", e.to_string().c_str());
	}
	catch (std::exception& e)
	{
		printf("error: %s\r\n", e.what());

		_critical("%s", e.what());
	}
	catch (...)
	{
		printf("unknown error!\r\n");

		_critical("unknown error!");
	}

	// signal error
	return 1;
}