#include <stdio.h>

#include "../core/utils/utils.h"
#include "../core/debug/evemon.h"
#include "../core/debug/exception.h"
#include "../core/net/net.h"

#include "server.h"

#include <signal.h>

// exit flag
thread::event g_exit;

void sigfunc(int sigval)
{
	// trigger exit signal
	g_exit.trigger();
}

// main program
int main(int argc, char** argv)
{
	// check number of arguments
	if (argc < 2)
	{
		printf("Usage: %s <address> [options]\r\n", argv[0]);
		printf("\r\n");
		printf("Options:\r\n");
		printf("\t-s use switching (unthreaded) mode\r\n");

		return -1;
	}

	// read arguments
	bool switching = false;

	for (int i = 2; i < argc; i++)
	{
		if (_strcmpi(argv[i], "-s") == 0)
		{
			switching = true;

			printf("Switching mode enabled.\r\n");
		}
		else
		{
			printf("unknown argument \"%s\"!\r\n", argv[i]);

			return -1;
		}
	}

	try
	{
		// initialize net interface
		net::init();

		// exit event on interrupt
		signal(SIGINT, &sigfunc);
		signal(SIGTERM, &sigfunc);

		// initialize server
		ping_server serve(argv[1]);

		// start server
		printf("Starting ping server...\r\n");

		if(!switching)
			serve.start();

		// wait for completion
		if (switching)
			g_exit.do_until([&](void)
				{
					if (!serve.sync_run())
						std::this_thread::sleep_for(std::chrono::milliseconds(1));
				});
		else
			g_exit.wait();

		// stop server
		printf("Stopping ping server...\r\n");

		if (!switching)
			serve.stop();

		// close net interface
		net::close();

		// return successfuly
		return 0;
	}
	catch (exception::base& e)
	{
		printf("error: %s\r\n", e.to_string().c_str());

		_critical("%s", e.to_string().c_str());
	}
	catch (std::exception& e)
	{
		printf("error: %s\r\n", e.what());

		_critical("%s", e.what());
	}
	catch (...)
	{
		printf("unknown error!\r\n");

		_critical("unknown error!");
	}

	// signal error
	return 1;
}