#include <stdio.h>

#include "../core/utils/utils.h"
#include "../core/debug/evemon.h"
#include "../core/debug/exception.h"
#include "../core/net/net.h"

#include "client.h"

#include <cmath>

// main program
int main(int argc, char** argv)
{
	// check number of arguments
	if (argc < 2)
	{
		printf("Usage: %s <address> [options]\r\n", argv[0]);
		printf("\r\n");
		printf("Options:\r\n");
		printf("\t-n<int> number of messages to send\r\n");

		return -1;
	}

	// read arguments
	size_t N = 4;

	for (int i = 2; i < argc; i++)
	{
		if (str_begins_with(argv[i], "-n"))
		{
			N = atol(argv[i] + 2);

			if (N == 0)
			{
				printf("Invalid number of samples!\r\n");

				return -1;
			}

			printf("using %zu samples.\r\n", N);
		}
		else
		{
			printf("unknown argument \"%s\"!\r\n", argv[i]);

			return -1;
		}
	}

	try
	{
		// initialize net interface
		net::init();

		// initialize server
		ping_client client(argv[1]);

		// start client
		printf("Pinging %s...\r\n", argv[1]);

		client.start();

		// send ping requests
		double sum = 0;
		double sumsq = 0;
		double nsum = 0;

		for (size_t i = 0; i < N; i++)
		{
			// reset event
			client.reset();

			// start monitoring
			auto tick1 = std::chrono::high_resolution_clock::now();

			// send ping request
			client.send(msg_ping_request());

			// wait for reply
			bool success = client.wait(1.0);

			if (!success)
			{
				printf("Reply not received within timeout period!\r\n");

				continue;
			}

			// compute latency
			auto tick2 = std::chrono::high_resolution_clock::now();
			auto delta_ticks = std::chrono::duration_cast<std::chrono::duration<double>>(tick2 - tick1);

			auto value = delta_ticks.count();

			nsum++;
			sum += value;
			sumsq += value * value;

			printf("Ping reply: %.3f ms\r\n", 1000.0 * value);
		}

		if (nsum > 0)
		{
			const double avg = sum / nsum;
			const double stdev = sqrt(sumsq / nsum - avg * avg);

			printf("\r\n\r\n");
			printf("Average Time: %.3f ms\r\n", 1000.0 * avg);
			printf("Standard Deviation: %.3f ms\r\n", 1000.0 * stdev);
		}

		// stop client
		client.stop();

		// close net interface
		net::close();

		// return successfuly
		return 0;
	}
	catch (exception::base& e)
	{
		printf("error: %s\r\n", e.to_string().c_str());

		_critical("%s", e.to_string().c_str());
	}
	catch (std::exception& e)
	{
		printf("error: %s\r\n", e.what());

		_critical("%s", e.what());
	}
	catch (...)
	{
		printf("unknown error!\r\n");

		_critical("unknown error!");
	}

	// signal error
	return 1;
}
