#pragma once

#include <string>

#include "../debug/exception.h"

#include "utils.h"

// singleton_exception class
class singleton_exception : public exception::base
{
public:
	singleton_exception(const std::string& type)
	{
		this->m_type = type;
	}

	virtual std::string to_string(void) const override
	{
		return std::string("Cannot access singleton instance of type \"") + this->m_type + std::string("\" !");
	}

private:
	std::string m_type;
};

// singleton class
template<class type> class singleton
{
public:
	static type* instance;

	static auto get_instance(void)
	{
		if (instance == nullptr)
			instance = new type();

		return instance;
	}

	// do not allow copy and move operators
	singleton<type>(const singleton<type>&) = delete;
	singleton<type>& operator=(const singleton<type>&) = delete;
	singleton<type>& operator=(const singleton<type>&&) = delete;

protected:
	// do not allow other creation than via get_instance
	singleton<type>(void) {}

private:
	// only main class can initialize
	void* operator new(size_t nSize)
	{
		return ::operator new(nSize);
	}
};

// automatically create and destroy singleton instance using RIAA idiom
template<class type> class auto_singleton
{
public:
	// create instance on initialization
	auto_singleton(void)
	{
		// create instance
		type::get_instance();
	}

	// destruct pointer on dtor
	~auto_singleton(void)
	{
		delete type::instance;
		type::instance = nullptr;
	}
};

// return instance or throw error
template<typename type> auto get_instance(void)
{
	auto p = type::get_instance();

	if (p == nullptr)
		throw_exception(singleton_exception, typeid(type).name());

	return p;
}

// initialize singleton automatically on program startup
#define INITIALIZE_SINGLETON_EX(type, name)	template<> type *singleton<type> :: instance = nullptr;									\
																															\
												auto_singleton<type> g_auto_singleton__##name;

#define INITIALIZE_SINGLETON(type)				INITIALIZE_SINGLETON_EX(type, type)
