#pragma once

#include "../utils/utils.h"

#include <chrono>
#include <mutex>
#include <condition_variable>

namespace thread
{
	// event class
	class event
	{
	public:

		// default constructor
		event(void)
		{
			this->m_flag = false;
		}

		// wait for the event, return false if timeout
		bool wait(double timeout = 0) const
		{
			std::unique_lock<std::mutex> lk(this->m_lock);

			if (timeout > 0)
			{
				return this->m_cv.wait_until(lk, std::chrono::system_clock::now() + std::chrono::duration<double, std::ratio<1>>(timeout), [&](void)
					{
						return this->m_flag;
					});
			}

			this->m_cv.wait(lk, [&](void)
				{
					return this->m_flag;
				});

			return true;
		}

		// trigger the event
		void trigger(void)
		{
			std::lock_guard<std::mutex> lk(this->m_lock);

			this->m_flag = true;

			this->m_cv.notify_all();
		}

		// reset the event
		void reset(void)
		{
			std::lock_guard<std::mutex> lk(this->m_lock);

			this->m_flag = false;

			this->m_cv.notify_all();
		}

		// return true if the event is pending
		bool is_pending(void) const
		{
			std::lock_guard<std::mutex> lk(this->m_lock);

			return !this->m_flag;
		}

		// perform action until states changes
		bool do_until(std::function<void(void)> func, double timeout = 0) const
		{
			auto tick1 = std::chrono::high_resolution_clock::now();

			// loop as long as event is not signalled
			while (is_pending())
			{
				// perform func
				if (func)
					func();

				if (timeout <= 0)
					continue;

				// get current time
				auto tick2 = std::chrono::high_resolution_clock::now();

				// compute time
				auto delta_ticks = std::chrono::duration_cast<std::chrono::duration<double>>(tick2 - tick1);

				// break if delta_ticks above timeout
				if (delta_ticks.count() > timeout)
					break;
			}

			// return triggered status
			return !is_pending();
		}

	private:
		mutable std::mutex m_lock;
		mutable std::condition_variable m_cv;

		bool m_flag;
	};
};