#pragma once

#include "../utils/utils.h"
#include "../debug/exception.h"
#include "../debug/evemon.h"
#include "../debug/exdebug.h"
#include "../debug/safe.h"
#include "../resources/file.h"
#include "../exception.h"

#include "exception.h"

#include <vector>
#include <string>
#include <ios>
#include <fstream>

namespace storage
{
	// manage buffer memory for storage
	class buffer
	{
	public:

		// initialize to null values
		buffer(void)
		{
			this->m_data = nullptr;
			this->m_size = 0;
		}

		// initialize with data
		buffer(const unsigned char* data, size_t size)
		{
			this->m_data = nullptr;
			this->m_size = 0;

			append(data, size, false);
		}

		// copy from buffer
		buffer(const buffer& other)
		{
			this->m_data = nullptr;
			this->m_size = 0;

			this->operator=(other);
		}

		// move from buffer
		buffer(buffer&& other)
		{
			this->m_data = nullptr;
			this->m_size = 0;

			this->operator=(std::move(other));
		}

		// release memory on dtor
		~buffer(void)
		{
			clear();
		}

		// copy operator
		buffer& operator=(const buffer& other)
		{
			// clear previous data
			clear();

			// copy data
			if (other.m_size != 0 && other.m_data != nullptr)
				append(other.m_data, other.m_size, false);

			return *this;
		}

		// move operator
		buffer& operator=(buffer&& other)
		{
			// clear previous data
			clear();

			// move new data
			this->m_data = other.m_data;
			this->m_size = other.m_size;

			other.m_data = nullptr;
			other.m_size = 0;

			return *this;
		}

		// allocate empty space
		size_t allocate(size_t bytes_size)
		{
			size_t new_size = MAKE_ADD(this->m_size, bytes_size);

			auto new_data = (unsigned char*)realloc(this->m_data, new_size);

			if (new_data == nullptr)
				throw_exception(memory_allocation_exception, new_size);

			this->m_data = new_data;

			memset(MAKE_ADDRESS(this->m_data, this->m_size), 0, bytes_size);

			size_t offset = this->m_size;
			this->m_size = new_size;

			return offset;
		}

		// allocate space for known type and return pointer to type
		template<typename type> size_t allocate(void)
		{
			return allocate(sizeof(type));
		}

		// cast to data at given offset
		template<typename type> type* ptr(size_t offset, size_t check_bytes = 0)
		{
			if (offset >= this->m_size)
				throw_exception(invalid_offset_exception);

			if (MAKE_ADD(offset, check_bytes) > this->m_size)
				throw_exception(type_fit_exception);

			return (type*)MAKE_ADDRESS(data(), offset);
		}

		// cast to data at given offset (const)
		template<typename type> const type* ptr(size_t offset, size_t check_bytes = 0) const
		{
			if (offset >= this->m_size && check_bytes != 0)
				throw_exception(invalid_offset_exception);

			if (MAKE_ADD(offset, check_bytes) > this->m_size)
				throw_exception(type_fit_exception);

			return (const type*)MAKE_ADDRESS(data(), offset);
		}

		// copy data
		size_t append(const unsigned char* mem, size_t bytes_size, bool recycle = false)
		{
			if (bytes_size == 0)
				return this->m_size;

			if (__ISNULLPTR(mem))
				throw_exception(null_pointer_exception);

			// check if data can be recycled
			if (recycle && this->m_size >= bytes_size)
			{
				// scan
				for (size_t i = 0; i <= MAKE_SUB(this->m_size, bytes_size); i++)
				{
					// return current position if data matches
					if (std::memcmp(MAKE_ADDRESS(this->m_data, i), mem, bytes_size) == 0)
						return i;
				}
			}

			// allocate new data otherelse
			size_t new_size = MAKE_ADD(this->m_size, bytes_size);

			auto new_data = (unsigned char*)realloc(this->m_data, new_size);

			if (new_data == nullptr)
				throw_exception(memory_allocation_exception, new_size);

			this->m_data = new_data;

			memcpy(MAKE_ADDRESS(this->m_data, this->m_size), mem, bytes_size);

			size_t offset = this->m_size;
			this->m_size = new_size;

			return offset;
		}

		// copy data of known type
		template<typename type> size_t append(const type& value, bool recycle=false)
		{
			return append((unsigned char*)&value, sizeof(value), recycle);
		}

		// copy string is special
		size_t append(const std::string& str, bool recycle=false)
		{
			return append((unsigned char*)str.c_str(), MAKE_ADD(str.length(), (size_t)1), recycle);
		}

		// copy other storage buffer is special
		size_t append(const storage::buffer& obj, bool recycle=false)
		{
			return append(obj.data(), obj.size(), recycle);
		}

		// pointer to data
		unsigned char* data(void)
		{
			return this->m_data;
		}

		// const pointer to data
		const unsigned char* data(void) const
		{
			return this->m_data;
		}

		// size of data
		size_t size() const
		{
			return this->m_size;
		}

		// return true if both blocks contains the same data
		bool equals(const buffer& other)
		{
			if (other.m_size != this->m_size)
				return false;

			if (__ISNULLPTR(other.m_data))
				return false;

			if (__ISNULLPTR(this->m_data))
				return false;

			return std::memcmp(this->m_data, other.m_data, this->m_size) == 0;
		}

		// clear buffer
		void clear(void)
		{
			if (this->m_data != nullptr)
				free(this->m_data);

			this->m_data = nullptr;
			this->m_size = 0;
		}

	private:
		unsigned char* m_data;
		size_t m_size;
	};
};
