#pragma once

#include "../utils/utils.h"
#include "../types/type.h"
#include "../debug/exception.h"

#include "exception.h"
#include "fs/object.h"

#include <vector>
#include <string>
#include <ios>
#include <fstream>

namespace storage
{
	// structure to hold storage list
	struct list_s
	{
		const char* owner;
		const char* name;
		const char* type;

		size_t bytes_size;
		size_t bytes_offset;

		bool is_object;
		size_t object_base_offset;
	};

	// base class
	class base
	{
	public:
		virtual ~base(void) {}

		virtual void push(storage::fs::object_base& obj) const = 0;
		virtual void pop(const storage::fs::object_base& obj) = 0;

		virtual std::string get_typename(void) const = 0;

		virtual void custom_push(storage::fs::object_base&) const { }
		virtual void custom_pop(const storage::fs::object_base&) { }

		void generic_push(storage::fs::object_base& obj, const void* base_address, struct list_s* list, size_t num_elems) const;
		void generic_pop(const storage::fs::object_base& obj, const void* base_address, struct list_s* list, size_t num_elems);
	};

	// SFINAE to check if storeable object
	template<typename type> struct is_base_object_t
	{
		static bool check(base*)
		{
			return true;
		}

		static bool check(...)
		{
			return false;
		}

		static bool test(void)
		{
			return check(static_cast<type*>(nullptr));
		}
	};

	template<typename type> static bool is_base_object(const type&)
	{
		return is_base_object_t<type> ::test();
	}

	// get base offset of multiple-inheritance classes
	template<class child, class parent> size_t get_base_ofs(void)
	{
		child tmp;

		parent* p = static_cast<parent*>(&tmp);

		return (size_t)((unsigned char*)p - (unsigned char*)&tmp);
	}

	// SFINAE for offset to storeable object
	template<typename type> struct object_base_offset_t
	{
		static size_t get_ofs(base*)
		{
			return get_base_ofs<type, base>();
		}

		static size_t get_ofs(...)
		{
			return 0;
		}

		static size_t get(void)
		{
			type tmp;

			return get_ofs(&tmp);
		}
	};

	template<typename type> size_t object_base_offset(const type&)
	{
		return object_base_offset_t<type> ::get();
	}

	// visual studio doesn't like typeid(((class*)0)->var)
	template<typename type> const char* get_var_typename(const type&)
	{
		return typeid(type).name();
	}
};

// macro to implement storage mechanisms
#define IMPLEMENT_STORAGE							static struct storage::list_s storage_data[];						\
																														\
													virtual void push(storage::fs::object_base& obj) const override;	\
													virtual void pop(const storage::fs::object_base& obj) override;		\
													virtual std::string get_typename(void) const override;


#define BEGIN_STORAGE(class)						struct storage::list_s class::storage_data[] = 


#define DECLARE_STORAGE(class, var)					{ #class, #var, storage::get_var_typename(((class*)0)->var), sizeof(class::var), (size_t)&((class*)0)->var, storage::is_base_object(((class*)0)->var), storage::object_base_offset(((class*)0)->var) }

#define END_STORAGE(class, parent)					;																					\
																																		\
													void class::push(storage::fs::object_base& obj) const								\
													{																					\
														if(strcmp(#class, #parent))														\
															parent :: push(obj);														\
																																		\
														size_t num_elems = sizeof(storage_data) / sizeof(struct storage::list_s);		\
																																		\
														generic_push(obj, this, storage_data, num_elems);								\
													}																					\
																																		\
													void class::pop(const storage::fs::object_base& obj)								\
													{																					\
														if(strcmp(#class, #parent))														\
															parent :: pop(obj);															\
																																		\
														size_t num_elems = sizeof(storage_data) / sizeof(struct storage::list_s);		\
																																		\
														generic_pop(obj, this, storage_data, num_elems);								\
													}																					\
																																		\
													std::string class::get_typename(void) const											\
													{																					\
														return #class;																	\
													}																					\
																																		\
													REGISTER_TYPENAME_EX(class, #class, class);

#define DECLARE_EMPTY_STORAGE(class, parent)		void class::push(storage::fs::object_base& obj) const								\
													{																					\
														if(strcmp(#class, #parent))														\
															parent :: push(obj);														\
																																		\
														generic_push(obj, this, nullptr, 0);											\
													}																					\
																																		\
													void class::pop(const storage::fs::object_base& obj)								\
													{																					\
														if(strcmp(#class, #parent))														\
															parent :: pop(obj);															\
																																		\
														generic_pop(obj, this, nullptr, 0);												\
													}																					\
																																		\
													std::string class::get_typename(void) const											\
													{																					\
														return #class;																	\
													}																					\
																																		\
													REGISTER_TYPENAME_EX(class, #class, class);
