#pragma once

#include "../debug/exdebug.h"

#include "../exception.h"

#include <memory>
#include <mutex>

namespace scope
{
	/* ticket class */
	class ticket
	{
	public:
		struct data_s
		{
			bool alive;
			mutable std::recursive_mutex lock;
		};

		/* handle class */
		class handle
		{
		public:

			// no default ctor
			handle(void) = delete;

			// copy as pointer on ctor
			handle(std::shared_ptr<const struct data_s> data)
			{
				this->m_data = data;
			}

			// lock object temporarily
			auto lock(void) const
			{
				if (__ISNULLPTR(this->m_data))
					throw_exception(null_pointer_exception);

				return std::lock_guard<std::recursive_mutex>(this->m_data->lock);
			}

			// check if object is still alive
			bool is_alive(void) const
			{
				if (__ISNULLPTR(this->m_data))
					return false;

				return this->m_data->alive;
			}

		private:
			std::shared_ptr<const struct data_s> m_data;
		};

	public:

		// allocate alive flag on constructor
		ticket(void)
		{
			_debug("acquiring scope ticket");

			this->m_data = std::make_shared<struct data_s>();
			
			if (!__ISNULLPTR(this->m_data))
				this->m_data->alive = true;
		}

		// no copy or move
		ticket(const ticket&) = delete;
		ticket(ticket&&) = delete;

		const ticket& operator=(const ticket&) = delete;
		const ticket& operator=(ticket&&) = delete;

		// release scope ticket on dtor
		~ticket(void)
		{
			_debug("releasing scope ticket");

			handle tmp(this->m_data);

			auto lk = tmp.lock();

			if (!__ISNULLPTR(this->m_data))
				this->m_data->alive = false;
		}

		// create handle
		auto create_handle(void) const
		{
			return handle(this->m_data);
		}

	private:
		std::shared_ptr<struct data_s> m_data;
	};
};