#pragma once

#include "../utils/utils.h"
#include "../debug/exdebug.h"
#include "../debug/evemon.h"
#include "../utils/singleton.h"
#include "../resources/uid.h"
#include "../scope/access.h"
#include "../scope/base.h"

#include "../exception.h"

#include "exception.h"

#include <string>
#include <unordered_map>
#include <list>

namespace message
{
	template<class msg_base> class entity;

	template<class msg_base> class manager : public singleton<manager<msg_base>>, public scope::base
	{
		friend class singleton<manager<msg_base>>;

		using entity_t = std::shared_ptr<entity<msg_base>>;

	public:
		class token : public scope::access<manager<msg_base>>
		{
		public:
			token(manager<msg_base>* parent, const scope::ticket& handle, const std::string uid) : scope::access<manager<msg_base>>(parent, handle)
			{
				this->m_uid = uid;
			}

			void unregister(void)
			{
				auto p = this->get();

				// unregister
				p->unregister(this->m_uid);

				// unregister mnemonic
				if (this->m_name.length() > 0)
					p->unregister_mnemonic(this->m_name);

				this->m_name = "";
			}

			const std::string& uid(void) const
			{
				return this->m_uid;
			}

			const std::string& name(void) const
			{
				return this->m_name;
			}

			void set_name(const std::string& name)
			{
				auto p = this->get();

				if(this->m_name.length() > 0)
					p->unregister_mnemonic(this->m_name);
			
				p->register_mnemonic(name, uid());

				this->m_name = name;
			}

			bool dispatch(const std::string& dest, const msg_base& msg) const
			{
				return this->get()->dispatch(uid(), dest, msg);
			}

		private:
			std::string m_uid;
			std::string m_name;
		};

	public:

		// add entity
		auto register_entity(entity_t obj)
		{
			if (__ISNULLPTR(obj))
				throw_exception(null_pointer_exception);

			auto uid = _create_uid();

			_debug("creating entity %s", uid.c_str());

			this->m_list.emplace(std::make_pair(uid, obj));

			return std::make_unique<token>(this, get_scope_ticket(), uid);
		}

		// inverse resolve of mnemonic
		std::string inv_resolve(const std::string& uid) const
		{
			for (auto& v : this->m_mnemonics)
				if (v.second == uid)
					return v.first;

			throw_exception(no_mnemonic_exception, uid);
		}

		// return true if name exists
		bool has(const std::string& name) const
		{
			return this->m_list.find(_get_uid(name)) != this->m_list.end();
		}

		// get list of all names
		std::list<std::string> all(const std::list<std::string>& excludes = {}) const
		{
			auto _exclude = [&](std::string uid)
			{
				for (auto& v : excludes)
					if (uid == _get_uid(v))
						return true;

				return false;
			};

			std::list<std::string> ret;

			for (auto& v : this->m_list)
				if (!_exclude(v.first))
					ret.push_back(v.first);

			return ret;
		}

	private:

		// unregister mnemonic
		void unregister_mnemonic(const std::string& name)
		{
			auto it = this->m_mnemonics.find(name);

			if (it == this->m_mnemonics.end())
				throw_exception(mnemonic_not_found_exception, name);
			
			this->m_mnemonics.erase(it);
		}

		// register mnemonic
		void register_mnemonic(const std::string& name, const std::string& uid)
		{
			if (has(name))
				throw_exception(name_already_exists_exception, name);

			_debug("registering \"%s\" as %s", name.c_str(), uid.c_str());

			this->m_mnemonics.emplace(std::make_pair(name, uid));
		}

		// unregister entity
		void unregister(const std::string& id)
		{
			// remove from list
			auto it = this->m_list.find(id);

			if (it != this->m_list.end())
				this->m_list.erase(it);
		}

		// dispatch message
		bool dispatch(const std::string& sender, const std::string& dest, const msg_base& msg) const
		{
			auto obj = _get(dest);

			if (__ISNULLPTR(obj))
				throw_exception(null_pointer_exception);

			return obj->handle(msg, sender);
		}

	private:
		manager(void) {}

		// create uid and check that it does not already exists
		std::string _create_uid(void) const
		{
			std::string uid;

			do
			{
				uid = resources::uid::create();
			} while (has(uid));

			return uid;
		}

		// transform name or uid into uid
		entity_t _get(const std::string& name) const
		{
			auto it = this->m_list.find(_get_uid(name));

			if (it == this->m_list.end())
				throw_exception(invalid_uid_exception, name);

			return it->second;
		}

		// get entity for specific name or uid
		std::string _get_uid(const std::string& name) const
		{
			auto it = this->m_mnemonics.find(name);

			if (it != this->m_mnemonics.end())
				return it->second;

			return name;
		}

		// members
		std::unordered_map<std::string, entity_t> m_list;
		std::unordered_map<std::string, std::string> m_mnemonics;
	};
};
