#pragma once

#include "../utils/utils.h"
#include "../debug/exdebug.h"

#include "base.h"

#include <list>
#include <functional>
#include <memory>
#include <string>

namespace message
{
	/* message_handler class */
	template<class base_type=base, typename ... params> class handler
	{
	private:
		/* message_listener_base class */
		class listener_base
		{
		public:
			virtual bool is_type(const base_type&) const = 0;
			virtual bool process(const base_type&, params ...) const = 0;
		};

		/* message_listener class */
		template<class specialized_type> class listener : public listener_base
		{
		public:
			using callback_t = std::function<void(const specialized_type&, params ...)>;

			// ctor
			listener(const callback_t& func) : m_callback(func)
			{
				static_assert(std::is_base_of<message::base, specialized_type>::value, "typename must be derived from message::base class");
			}

			// return true if same type
			virtual bool is_type(const base_type& msg) const override
			{
				return is_type_name<specialized_type>(msg.get_type_name());
			}

			// try to process message
			virtual bool process(const base_type& msg, params ... args) const override
			{
				if (!is_type(msg))
					return false;

				this->m_callback(dynamic_cast<const specialized_type&>(msg), args ...);

				return true;
			}

		private:
			callback_t m_callback;
		};

	public:

		// handle incomming message
		bool handle(const base_type& msg, params ... args)
		{
			bool ret = false;

			for (auto& p : this->m_listeners)
			{
				if (p == nullptr)
					continue;

				ret |= p->process(msg, args ...);
			}

			return ret;
		}

	protected:

		// register handler
		template<class specialized_type> void register_listener(const typename listener<specialized_type>::callback_t& func)
		{
			this->m_listeners.emplace_back(std::make_unique< listener<specialized_type> >(func));
		}

	private:
		// list of handlers
		std::list< std::unique_ptr<listener_base>> m_listeners;
	};
};

#define REGISTER_MESSAGE_EX(class, type, func)		register_listener<type>(std::bind(&class::func, this, std::placeholders::_1));
#define REGISTER_MESSAGE(class, type)				REGISTER_MESSAGE_EX(class, type, recv_##type);

#define HANDLE_MESSAGE_EX(type, obj, func)			void func(const type& obj)
#define HANDLE_MESSAGE(type, obj)					HANDLE_MESSAGE_EX(type, obj, recv_##type)