#pragma once

#include "../utils/utils.h"
#include "../debug/exception.h"
#include "../debug/evemon.h"
#include "../debug/exdebug.h"
#include "../utils/singleton.h"
#include "../threads/lock.h"

#include "exception.h"

#include <map>
#include <unordered_map>
#include <list>

/* memory_manager singleton class */
class memory_manager : public singleton<memory_manager>
{
	friend class singleton<memory_manager>;

public:
	// destructor
	~memory_manager(void)
	{
		// purge free list
		purge();

		// report error if any locked memory
		for(auto& v : this->m_locked)
			_error("memory leak %zu bytes!", v.second);
	}

	// acquire memory
	void* acquire(size_t bytes_size)
	{
		if (__ISZERO(bytes_size))
			return nullptr;

		// lock mutex
		AUTOLOCK(this->m_lock);

		void* p = nullptr;

		// get list for that size
		auto& list = get(bytes_size);

		// create element if any
		if (list.empty())
		{
			p = _aligned_malloc(bytes_size, 32);

			_debug("Allocating %zu bytes", bytes_size);
		}
		else
		{
			// get data
			auto it = list.begin();
			p = *it;

			// remove element
			list.erase(it);
		}

		// add to list
		this->m_locked.insert(std::make_pair(p, bytes_size));

		// return pointer
		return p;
	}

	// release memory
	void release(void* p)
	{
		if (p == nullptr)
			return;

		// lock mutex
		AUTOLOCK(this->m_lock);

		// lookup in locked vector
		auto it = this->m_locked.find(p);

		// throw error if not found
		if (it == this->m_locked.end())
			throw_exception(page_not_found_exception);

		// transfer data to free list
		auto& list = get(it->second);

		list.push_back(it->first);
		
		// remove from locked list
		this->m_locked.erase(it);
	}

	// purge expired
	void purge(void)
	{
		// lock mutex
		AUTOLOCK(this->m_lock);

		// browse free list
		for (auto& v : this->m_free)
		{
			for (auto& w : v.second)
				if (w != nullptr)
				{
					_aligned_free(w);
					w = nullptr;
				}
		}

		// clear free list
		this->m_free.clear();
	}

private:

	// constructor
	memory_manager(void) {}

	// get list of free memory
	std::list<void*>& get(size_t nSize)
	{
		// search in map
		auto it = this->m_free.find(nSize);

		// return if found
		if (it != this->m_free.end())
			return it->second;

		// otherelse create one
		auto tmp = this->m_free.insert(std::make_pair(nSize, std::list<void*>()));

		// return value
		return tmp.first->second;
	}

	// lock
	std::mutex m_lock;

	// member variable are distributed as list of free and locked
	std::unordered_map<size_t, std::list<void*>> m_free;
	std::unordered_map<void*, size_t> m_locked;
};
