#pragma once

#include "../utils/utils.h"
#include "../debug/evemon.h"
#include "../debug/exdebug.h"
#include "../utils/singleton.h"
#include "../threads/lock.h"
#include "../threads/thread.h"

#include "tpool.h"

#include "exception.h"
#include "jobs.h"

/* Jobs Manager class */
class jobs_manager : public singleton<jobs_manager>, public thread::base
{
public:
	friend class singleton<jobs_manager>;

	// clean exit
	virtual ~jobs_manager(void)
	{
		try
		{
			stop();
		}
		catch (...) {}

		this->m_jobs.clear();
	}

	// check if empty
	bool empty(void)
	{
		AUTOLOCK(this->m_lock);

		return this->m_jobs.empty();
	}

	// add job
	void queue(std::shared_ptr<job_base> p)
	{
		_debug("queuing job %p", p);

		// skip null jobs
		if (__ISNULLPTR(p))
			throw_exception(null_job_exception);

		// initialize job first if not already done
		if (p->get_status() == job_base::status::init_required)
			p->init();

		// skip jobs that are not pending
		if (p->get_status() != job_base::status::waiting)
			throw_exception(job_queue_exception);

		// push jobs to queue
		{
			
			AUTOLOCK(this->m_lock);

			p->push(this->m_jobs);
		}

		// sync mode
		if (!is_running())
		{
			// performs jobs as long as list is not empty
			while (!empty())
				run();
		}
	}

private:

	// ctor
	jobs_manager(void) {}

	// inner queue
	void disptach(std::shared_ptr<job_base> p);

	// run
	virtual void run(void)
	{
		// pick up a job
		auto p = pop();

		// wait 10 ms if stack is empty
		if (p == nullptr)
		{
			sleep(10e-3);

			return;
		}

		// assign to thread pool manager
		disptach(p);
	}

	// pick a job
	std::shared_ptr<job_base> pop(void)
	{
		AUTOLOCK(this->m_lock);

		// browse list
		for (auto p : this->m_jobs)
		{
			// skip null
			if (__ISNULLPTR(p))
				continue;

			// skip if not ready to accept
			if (p->accept() != job_base::ready::accept)
				continue;

			// adopt if waiting and remove from list
			if(p->get_status() == job_base::status::waiting)
				p->adopt();

			this->m_jobs.remove(p);

			// return job
			return p;
		}

		// no jobs to process
		return nullptr;
	}

	// mutex
	lock_t m_lock;

	// list of jobs
	std::list<std::shared_ptr<job_base>> m_jobs;
};

// wait for
static void wait_for(std::list<std::shared_ptr<job_base>> list)
{
	_debug("waiting for %zu jobs", list.size());

	// temporarily increase number of allowed threads
	auto obj = ::get_instance<thread_pool_manager>()->allow_more();

	// queue jobs
	for (auto& p : list)
		get_instance<jobs_manager>()->queue(p);


	// wait for job to finish
	for (auto& p : list)
	{
		if (p != nullptr)
			p->wait();
	}

	// eventually throw error
	for (auto& p : list)
	{
		if (__ISNULLPTR(p))
			continue;
		
		// throw error
		if(p->get_status() == job_base::status::error)
			throw_exception(job_exception, p->get_error_message());

		// job should be finished by now
		if (p->get_status() != job_base::status::done)
			throw_exception(job_done_exception);
	}
}
