#pragma once

#include "../utils/utils.h"
#include "../debug/evemon.h"
#include "../debug/exdebug.h"
#include "../threads/event.h"

#include "exception.h"

#include <memory>
#include <atomic> 
#include <list>

/* job_base interface class */
class job_base : public std::enable_shared_from_this<job_base>
{
public:
	// possible status of jobs
	enum class status
	{
		init_required,
		waiting,
		adopted,
		running,
		done,
		error,
		cancelled,
	};

	// possible ready status
	enum class ready
	{
		accept,
		decline,
	};

	// default ctor
	job_base(void)
	{
		this->m_status = status::init_required;
	}

	// initialize jobs
	virtual void init(void)
	{
		_debug("initializing job %p", this);

		// initialize dependencies first
		for (auto& p : this->m_dependencies)
		{
			if (__ISNULLPTR(p))
				continue;
			
			p->init();
		}

		// call inner init
		_init();

		// set status as pending
		set_status(status::waiting);
	}

	// push tree to list
	void push(std::list<std::shared_ptr<job_base>>& list)
	{
		for (auto& p : this->m_dependencies)
		{
			if (__ISNULLPTR(p))
				continue;

			p->push(list);
		}

		list.push_back(shared_from_this());
	}

	// get current status
	auto get_status(void) const
	{
		return this->m_status.load();
	}

	// return error message
	std::string get_error_message(void) const
	{
		return this->m_error;
	}

	// get ready status
	auto accept(void) const
	{
		/*
		// only accept if job is pending
		if (get_status() != status::waiting)
			return ready::decline;
		*/

		// decline if not all dependencies are done
		for (auto& v : this->m_dependencies)
			if (v != nullptr && v->get_status() != status::done)
				return ready::decline;

		// ready to process
		return ready::accept;
	}

	// adopt job
	void adopt(void)
	{
		_debug("adopting job %p", this);

		if (get_status() != status::waiting)
			throw_exception(job_adopt_exception);

		set_status(status::adopted);
	}

	// release job
	void release(void)
	{
		_debug("releasing job %p", this);

		if (get_status() != status::adopted)
			throw_exception(job_release_exception);

		set_status(status::waiting);
	}

	// set done action
	void set_done_action(std::function<void(void)> func)
	{
		this->m_done_action = func;
	}

	// generic run
	void run(void)
	{
		_debug("running job %p", this);

		// skip if not waiting
		switch (get_status())
		{
			case status::waiting:
			case status::adopted:
				break;

			default:
				_warning("unable to run job %p!", this);

				return;
		}

		// mark as running
		set_status(status::running);

		this->m_done.reset();

		try
		{
			// run
			_run();

			// mark as done
			set_status(status::done);

			// call done action
			if(this->m_done_action)
				this->m_done_action();
		}
		catch (const exception::base& e)
		{
			// set error status
			set_status(status::error);

			// clone error message
			this->m_error = e.to_string();

			_error("job %p triggered an exception: %s", this, e.to_string().c_str());
		}
		catch (const std::exception& e)
		{
			// set error status
			set_status(status::error);

			// clone error message
			this->m_error = e.what();

			_error("job %p triggered a std exception: %s", this, e.what());
		}
		catch (...)
		{
			// set error status
			set_status(status::error);

			// create empty error message
			this->m_error = "unknown exception";

			_error("job %p triggered an unknown exception", this);
		}

		// trigger done event
		this->m_done.trigger();

		_debug("job %p done", this);
	}

	// wait for job
	auto wait(double timeout=0) const
	{
		if (!this->m_done.is_pending())
			return true;

		return this->m_done.wait(timeout);
	}

	// reset job
	virtual void reset(void)
	{
		_debug("reseting job %p", this);

		// wait for job to finish
		wait();

		// reset done event
		this->m_done.reset();

		// set status as pending init
		set_status(status::init_required);

		// eventually clear errors
		this->m_error = "";
	}

protected:

	// run procedure
	virtual void _run(void) = 0;

	// init procedure
	virtual void _init(void) {}

	// add job to dependencies
	void add_dependency(std::shared_ptr<job_base> p)
	{
		_debug("adding %p as a dependency to job %p", p, this);

		// do not accept null jobs
		if (__ISNULLPTR(p))
			throw_exception(null_job_exception);

		// only accept jobs while in init state
		if (get_status() != status::init_required)
			throw_exception(job_add_exception);

		// add dependency
		this->m_dependencies.push_back(p);
	}

	// set current status
	void set_status(status e)
	{
		this->m_status.store(e);
	}

private:
	std::string m_error;
	std::list<std::shared_ptr<job_base>> m_dependencies;
	std::atomic<status> m_status;
	std::function<void(void)> m_done_action;

	thread::event m_done;
};
