#pragma once

#include "../storage/buffer.h"
#include "../types/endian.h"

// return SHA256 hash of input buffer
static storage::buffer sha256(storage::buffer in)
{
	// algorithm is based on 32 bits unsigned integers
	using type = uint32_t;

	// make endianess correct
	auto _endian32 = [](uint32_t val)
	{
		if constexpr (!endian::is_little_endian())
			return val;

		auto a = val & 0xFF;
		auto b = (val >> 8) & 0xFF;
		auto c = (val >> 16) & 0xFF;
		auto d = val >> 24;

		return (a << 24) | (b << 16) | (c << 8) | d;
	};

	auto _endian64 = [](uint64_t val)
	{
		if constexpr (!endian::is_little_endian())
			return val;

		auto a = val & 0xFF;
		auto b = (val >> 8) & 0xFF;
		auto c = (val >> 16) & 0xFF;
		auto d = (val >> 24) & 0xFF;
		auto e = (val >> 32) & 0xFF;
		auto f = (val >> 40) & 0xFF;
		auto g = (val >> 48) & 0xFF;
		auto h = val >> 56;

		return (a << 56) | (b << 48) | (c << 40) | (d << 32) | (e << 24) | (f << 16) | (g << 8) | h;
	};

	// inner functions that are used
	auto _ch = [](type x, type y, type z)
	{
		return (x & y) ^ ((~x) & z);
	};

	auto _maj = [](type x, type y, type z)
	{
		return (x & y) ^ (x & z) ^ (y & z);
	};

	auto _shr = [](type x, unsigned int n)
	{
		return x >> n;
	};

	auto _rotr = [](type x, unsigned int n)
	{
		return (x >> n) | (x << (32 - n));
	};

	auto _lsigma0 = [=](type x)
	{
		return _rotr(x, 7) ^ _rotr(x, 18) ^ _shr(x, 3);
	};

	auto _lsigma1 = [=](type x)
	{
		return _rotr(x, 17) ^ _rotr(x, 19) ^ _shr(x, 10);
	};

	auto _bsigma0 = [=](type x)
	{
		return _rotr(x, 2) ^ _rotr(x, 13) ^ _rotr(x, 22);
	};

	auto _bsigma1 = [=](type x)
	{
		return _rotr(x, 6) ^ _rotr(x, 11) ^ _rotr(x, 25);
	};

	// padd message to 512 bits
	auto size = in.size() * 8;
	auto padding = ((int64_t)448 - (int64_t)(size + 1)) % (int64_t)512;

	if (padding < 0)
		padding += 512;

	in.append<unsigned char>(0x80, false);

	for (size_t i = 1; i < (size_t)(padding + 1) / 8; i++)
		in.append<unsigned char>(0, false);

	in.append<uint64_t>(_endian64((uint64_t)size));

	// initialize
	type H[] = { 0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a, 0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19 };
	type K[] = { 0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5, 0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5, 0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3, 0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174, 0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc, 0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da, 0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7, 0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967, 0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13, 0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85, 0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3, 0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070, 0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5, 0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3, 0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208, 0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2 };

	auto pWords = in.ptr<type>(0);

	// digest message
	for (size_t i = 0; i < in.size() / sizeof(type); i += 16)
	{
		// load all words
		type M[16];

		for (size_t j = 0; j < 16; j++)
			M[j] = _endian32(pWords[i + j]);

		// build W table
		type W[64];

		for (size_t t = 0; t < 16; t++)
			W[t] = M[t];

		for (size_t t = 16; t < 64; t++)
			W[t] = _lsigma1(W[t - 2]) + W[t - 7] + _lsigma0(W[t - 15]) + W[t - 16];
	
		// copy H from previous rounds
		auto a = H[0];
		auto b = H[1];
		auto c = H[2];
		auto d = H[3];
		auto e = H[4];
		auto f = H[5];
		auto g = H[6];
		auto h = H[7];

		// perform some computing
		for (size_t t = 0; t < 64; t++)
		{
			auto T1 = h + _bsigma1(e) + _ch(e, f, g) + K[t] + W[t];
			auto T2 = _bsigma0(a) + _maj(a, b, c);
			h = g;
			g = f;
			f = e;
			e = d + T1;
			d = c;
			c = b;
			b = a;
			a = T1 + T2;
		}

		// update hash values
		H[0] += a;
		H[1] += b;
		H[2] += c;
		H[3] += d;
		H[4] += e;
		H[5] += f;
		H[6] += g;
		H[7] += h;
	}

	// concatanate hash values
	storage::buffer hash;

	for (size_t i = 0; i < 8; i++)
		if constexpr (endian::is_little_endian())
			hash.append<type>(H[7 - i]);
		else
			hash.append<type>(H[i]);

	return hash;
}

// empty SHA-256
static storage::buffer sha256(void)
{
	return sha256(storage::buffer());
}

// shorthand notation for SHA-256 when given raw pointer
static storage::buffer sha256(unsigned char* pData, size_t nSize)
{
	return sha256(storage::buffer(pData, nSize));
}

// shorthand notation for strings
static storage::buffer sha256(const std::string& rString)
{
	return sha256((unsigned char*)rString.c_str(), rString.length());
}

// shorthand notation for C strings
static storage::buffer sha256(const char *pszString)
{
	if (pszString == nullptr)
		return sha256(storage::buffer());

	return sha256((unsigned char*)pszString, strlen(pszString));
}