#include <stdio.h>
#include <filesystem>

#include "../core/utils/utils.h"
#include "../core/debug/evemon.h"
#include "../core/debug/exdebug.h"
#include "../core/debug/exception.h"
#include "../core/library/library.h"
#include "../core/library/interface/interface.h"
#include "../core/storage/file.h"

#include "../common/common.h"

/* sample_job job class */
class sample_job : public job_base
{
public:
	sample_job(const double value)
	{
		this->m_out = 0;
		this->m_value = value;
	}

	const double get(void) const
	{
		return this->m_out;
	}

private:

	// nothing to do
	virtual void _run(void) override
	{
		this->m_out = 2.0 * this->m_value;
	}

	double m_value, m_out;
};

// queue job
void queue_job(std::shared_ptr<job_base> p)
{
	library::interface::get<job_manager_interface>()->queue(p);
}

// main program
int main(int argc, char** argv)
{
	try
	{
		// load libraries
		for (auto const& curr_entry : std::filesystem::directory_iterator("./"))
		{
			if (!curr_entry.is_regular_file())
				continue;

			auto filename = curr_entry.path().string();

#if _WIN32
			if (_strcmpi(split_file_parts(filename).ext.c_str(), "dll") != 0)
				continue;
#else
			if (_strcmpi(split_file_parts(filename).ext.c_str(), "so") != 0)
				continue;
#endif

			get_instance<library::manager>()->load(filename);
		}

		// TODO:
		//	- export core version from library to test

		/* example 1 */
		{
			// set current interface
			library::interface::set<job_manager_interface>("local job manager");

			// queue sample job
			auto p = std::make_shared<sample_job>(10);

			queue_job(p);

			p->wait();

			printf("value: %f\r\n", p->get());
		}

		/* example 2 */
		{
			// create instance
			auto p = create_instance<remote_module>("local job manager/sample_module");

			// set value
			p->set(5);

			// print value
			printf("value = %d\r\n", p->get());
		}

		/* example 3 */
		{
			auto func = (void(*)(void))get_instance<library::manager>()->get_addr("local job manager", "some_work");

			(*func)();
		}

		/* example 4 */
		{
			// save
			{
				// create instance
				auto p = create_instance<remote_module>("local job manager/sample_module");

				// set value
				p->set(10);

				// storage object
				storage::container container;
				container.push_back(p);

				storage::save_buffer_to_file("temp.bin", container.pack('SPC0'));
			}

			// reload
			{
				storage::container container;

				container.unpack(storage::load_buffer_from_file("temp.bin"), [](const std::string& name)
					{
						return create_instance<remote_module>(name);
					});

				if (container.size() > 0)
				{
					std::shared_ptr<storage::base> p;
					std::string owner, name;

					std::tie(owner, name, p) = container.front();

					// print value
					printf("value = %d\r\n", std::static_pointer_cast<remote_module>(p)->get());
				}
				else
					printf("no data!\r\n");
			}
		}
	
		// return successfuly
		return 0;
	}
	catch (exception::base& e)
	{
		printf("error: %s\r\n", e.to_string().c_str());

		_critical("%s", e.to_string().c_str());
	}
	catch (std::exception& e)
	{
		printf("error: %s\r\n", e.what());

		_critical("%s", e.what());
	}
	catch (...)
	{
		printf("unknown error!\r\n");

		_critical("unknown error!");
	}

	// signal error
	return 1;
}