#pragma once

#include "../utils/utils.h"
#include "../debug/exception.h"

#include "event.h"

#include <atomic>
#include <thread>

// default time for normal closing before forcing
#define THREAD_CLOSE_TIMEOUT_DELAY		10.0

namespace thread
{
	// base
	class base
	{
	public:

		// default constructor
		base(void)
		{
			this->m_quit = false;

			this->m_done.trigger();
		}

		// prevent move
		base(base&&) = delete;
		const base& operator=(base&&) = delete;

		// prevent copy
		base(const base&) = delete;
		const base& operator=(const base&) = delete;

		// default destructor
		virtual ~base(void)
		{
			_debug("destroying thread %p...", this);

			// stop on quit
			try
			{
				stop();
			}
			catch (...) {}
		}

		// wait for thread
		bool wait(double delay = 0) const
		{
			return this->m_done.wait(delay);
		}

		// start thread
		void start(void)
		{
			// skip if already started
			if (is_running())
				return;

			// stop in case it was not cleaned properly
			stop();

			// reset quit var
			this->m_quit = false;

			this->m_done.reset();

			_debug("starting thread %p...", this);

			// create thread
			this->m_thread = std::thread(std::bind(&base::loop, this));

			// call on_start
			on_start();
		}

		// stop thread
		void stop(void)
		{
			_debug("stopping thread %p...", this);

			// skip if thread is not joinable
			if (!this->m_thread.joinable())
				return;

			// set quit to true
			this->m_quit = true;

			wait();

			// join thread
			this->m_thread.join();

			// call on_stop
			on_stop();
		}

		// return true if thread is running
		bool is_running(void) const
		{
			return this->m_done.is_pending();
		}

		// sleep
		void sleep(double delay)
		{
			std::this_thread::sleep_for(std::chrono::duration<double, std::ratio<1>>(delay));
		}

	protected:

		// quit condition
		virtual bool stop_condition(void) const
		{
			return false;
		}

		// every thread MUST implement this function
		virtual void run(void) = 0;

		// called after thread has started
		virtual void on_start(void) {}

		// called after thread has stopped
		virtual void on_stop(void) {}

	private:

		// internal loop
		void loop(void)
		{
			_debug("start thread loop %p", this);

			// call run function as long as quit is not set to true
			try
			{
				while (!this->m_quit && !stop_condition())
					run();
			}
			catch (...)
			{
				_error("thread stopped due to an error!\r\n");
			}

			this->m_done.trigger();

			_debug("thread %p has finished", this);
		}

		std::thread m_thread;
		event m_done;

		volatile std::atomic<bool> m_quit;
	};
};