#pragma once

#include "../../../utils/utils.h"
#include "../../../debug/exception.h"
#include "../../../debug/evemon.h"
#include "../../../debug/exdebug.h"
#include "../../../debug/safe.h"
#include "../../../types/endian.h"

#include "../object.h"

#include <vector>
#include <string>
#include <list>
#include <memory>

namespace storage::fs::spc0
{
	struct var_s : public storage::fs::var_base_s
	{
		size_t bytes_size;

		void* data;
	};

	// object class
	class object : public storage::fs::object_base
	{
	public:
		using storage::fs::object_base::object_base;

		// add variable to list
		virtual void push_var(const std::string& owner, const std::string& name, const std::string& type, size_t bytes_size, void* mem) override
		{
			auto p = std::make_shared<var_s>();
			
			p->owner = owner;
			p->name = name;
			p->type = type;

			p->bytes_size = bytes_size;

			if (p->bytes_size > 0)
			{
				p->data = malloc(p->bytes_size);

				if (p->data == nullptr)
					throw_exception(memory_allocation_exception, p->bytes_size);

				//endian::write(type, mem, s.data, s.bytes_size);
				memcpy(p->data, mem, p->bytes_size);
			}
			else
				p->data = nullptr;

			add_variable(p);
		}

		// read variable from list
		virtual bool pop_var(const std::string& owner, const std::string& name, const std::string& type, size_t bytes_size, void* mem) const override
		{
			// browse list
			auto vars = _get_vars();

			for (auto& v : vars)
			{
				auto p = std::dynamic_pointer_cast<var_s>(v);

				if (p == nullptr)
					continue;

				// skip if different
				if (p->name != name || p->owner != owner || p->type != type)
					continue;

				// skip if size doesn't match
				if (p->bytes_size != bytes_size)
					throw_exception(bytesize_not_matching_exception, p->owner + std::string("::") + p->name, p->type);

				// read type
				// endian::read(v.type, v.data, mem, v.bytes_size);
				memcpy(mem, p->data, p->bytes_size);

				// type was read
				return true;
			}

			// return false if not found
			return false;
		}

		// pack data into a buffer
		virtual buffer pack(void) const override
		{
			// buffers
			storage::buffer strings, vars, data, children, out;

			// declare header
			size_t hdr_ofs = out.allocate<struct pack_header_s>();

			out.ptr<struct pack_header_s>(hdr_ofs)->num_vars = 0;
			out.ptr<struct pack_header_s>(hdr_ofs)->num_children = 0;

			out.ptr<struct pack_header_s>(hdr_ofs)->owner_ofs = htof(MAKE_UINT64(strings.append(this->get_owner_name(), true)));
			out.ptr<struct pack_header_s>(hdr_ofs)->name_ofs = htof(MAKE_UINT64(strings.append(this->get_var_name(), true)));
			out.ptr<struct pack_header_s>(hdr_ofs)->typename_ofs = htof(MAKE_UINT64(strings.append(this->get_type_name(), true)));

			// browse all vars
			size_t num_vars = 0;

			auto var_objects = get_vars();

			for (auto& v : var_objects)
			{
				auto p = std::dynamic_pointer_cast<const var_s>(v);

				if (p == nullptr)
					continue;

				auto curr = vars.ptr<struct pack_var_s>(vars.allocate<struct pack_var_s>());

				curr->owner_ofs = htof(MAKE_UINT64(strings.append(p->owner, true)));
				curr->name_ofs = htof(MAKE_UINT64(strings.append(p->name, true)));
				curr->type_ofs = htof(MAKE_UINT64(strings.append(p->type, true)));

				curr->data_ofs = htof(MAKE_UINT64(data.append((unsigned char*)p->data, p->bytes_size, false)));
				curr->bytes_size = htof(MAKE_UINT64(p->bytes_size));

				MAKE_INC(num_vars, (size_t)1);
			}

			out.ptr<struct pack_header_s>(hdr_ofs)->num_vars = htof(MAKE_UINT64(num_vars));

			// browse all children
			size_t num_children = 0;

			auto child_objects = this->get_children();

			for (auto p : child_objects)
			{
				if (p == nullptr)
					continue;

				buffer sub_buffer = p->pack();

				auto curr = children.ptr<struct pack_child_s>(children.allocate<struct pack_child_s>());

				curr->owner_ofs = htof(MAKE_UINT64(strings.append(p->get_owner_name(), true)));
				curr->name_ofs = htof(MAKE_UINT64(strings.append(p->get_var_name(), true)));

				curr->data_ofs = htof(MAKE_UINT64(data.append(sub_buffer, false)));
				curr->size = htof(MAKE_UINT64(sub_buffer.size()));

				MAKE_INC(num_children, (size_t)1);
			}

			out.ptr<struct pack_header_s>(hdr_ofs)->num_children = htof(MAKE_UINT64(num_children));

			// add temporary buffers to final buffer
			out.ptr<struct pack_header_s>(hdr_ofs)->strings.offset = htof(MAKE_UINT64(out.append(strings, false)));
			out.ptr<struct pack_header_s>(hdr_ofs)->strings.size = htof(MAKE_UINT64(strings.size()));

			out.ptr<struct pack_header_s>(hdr_ofs)->vars.offset = htof(MAKE_UINT64(out.append(vars, false)));
			out.ptr<struct pack_header_s>(hdr_ofs)->vars.size = htof(MAKE_UINT64(vars.size()));

			out.ptr<struct pack_header_s>(hdr_ofs)->children.offset = htof(MAKE_UINT64(out.append(children, false)));
			out.ptr<struct pack_header_s>(hdr_ofs)->children.size = htof(MAKE_UINT64(children.size()));

			out.ptr<struct pack_header_s>(hdr_ofs)->data.offset = htof(MAKE_UINT64(out.append(data, false)));
			out.ptr<struct pack_header_s>(hdr_ofs)->data.size = htof(MAKE_UINT64(data.size()));

			// return buffer
			return out;
		}

		// unpack data
		void unpack(const buffer& obj)
		{
			// clear all first to avoid duplications
			clear();

			// get header
			auto hdr = obj.ptr<struct pack_header_s>(0, sizeof(struct pack_header_s));

			// get strings buffer
			auto strings_offset = MAKE_SIZE_T(ftoh(hdr->strings.offset));
			auto strings_size = MAKE_SIZE_T(ftoh(hdr->strings.size));

			auto strings = obj.ptr<char>(strings_offset, strings_size);

			// get vars buffer
			auto vars_offset = MAKE_SIZE_T(ftoh(hdr->vars.offset));
			auto num_vars = MAKE_SIZE_T(ftoh(hdr->num_vars));
			auto vars_size = MAKE_SIZE_T(ftoh(hdr->vars.size));

			if (vars_size != MAKE_MULT(sizeof(struct pack_var_s), num_vars))
				throw_exception(buffer_read_exception);

			auto vars = obj.ptr<struct pack_var_s>(vars_offset, vars_size);

			// get children
			auto children_offset = MAKE_SIZE_T(ftoh(hdr->children.offset));
			auto num_children = MAKE_SIZE_T(ftoh(hdr->num_children));
			auto children_size = MAKE_SIZE_T(ftoh(hdr->children.size));

			if (children_size != MAKE_MULT(sizeof(struct pack_child_s), num_children))
				throw_exception(buffer_read_exception);

			auto children = obj.ptr<struct pack_child_s>(children_offset, children_size);

			// get data buffer
			auto data_offset = MAKE_SIZE_T(ftoh(hdr->data.offset));
			auto data_size = MAKE_SIZE_T(ftoh(hdr->data.size));

			auto data = obj.ptr<unsigned char>(data_offset, data_size);

			// read data
			auto name_offset = MAKE_SIZE_T(ftoh(hdr->name_ofs));
			auto owner_offset = MAKE_SIZE_T(ftoh(hdr->owner_ofs));
			auto type_offset = MAKE_SIZE_T(ftoh(hdr->typename_ofs));

			assert_string(strings, name_offset, strings_size);
			assert_string(strings, owner_offset, strings_size);
			assert_string(strings, type_offset, strings_size);

			this->set_var_name(MAKE_ADDRESS(strings, name_offset));
			this->set_owner_name(MAKE_ADDRESS(strings, owner_offset));
			this->set_type_name(MAKE_ADDRESS(strings, type_offset));

			for (size_t i = 0; i < num_vars; i++)
			{
				auto& curr = vars[i];

				auto name_offset = MAKE_SIZE_T(ftoh(curr.name_ofs));
				auto owner_offset = MAKE_SIZE_T(ftoh(curr.owner_ofs));
				auto type_offset = MAKE_SIZE_T(ftoh(curr.type_ofs));
				auto data_offset = MAKE_SIZE_T(ftoh(curr.data_ofs));

				auto bytes_size = MAKE_SIZE_T(ftoh(curr.bytes_size));

				assert_string(strings, owner_offset, strings_size);
				assert_string(strings, name_offset, strings_size);
				assert_string(strings, type_offset, strings_size);

				auto name = MAKE_ADDRESS(strings, name_offset);
				auto owner = MAKE_ADDRESS(strings, owner_offset);
				auto type = MAKE_ADDRESS(strings, type_offset);
				auto mem = MAKE_ADDRESS(data, data_offset);

				push_var(owner, name, type, bytes_size, (void*)mem);
			}

			for (size_t i = 0; i < MAKE_SIZE_T(ftoh(hdr->num_children)); i++)
			{
				auto& curr = children[i];

				auto owner_offset = MAKE_SIZE_T(ftoh(curr.owner_ofs));
				auto name_offset = MAKE_SIZE_T(ftoh(curr.name_ofs));
				auto data_offset = MAKE_SIZE_T(ftoh(curr.data_ofs));

				assert_string(strings, owner_offset, strings_size);
				assert_string(strings, name_offset, strings_size);

				auto name = MAKE_ADDRESS(strings, name_offset);
				auto owner = MAKE_ADDRESS(strings, owner_offset);
				auto mem = MAKE_ADDRESS(data, data_offset);
				auto mem_size = MAKE_SIZE_T(ftoh(curr.size));

				auto sub_object = create_sub_object(owner, name);

				if (MAKE_ADD(data_offset, mem_size) > data_size)
					throw_exception(buffer_read_exception);

				buffer temp_buffer;

				temp_buffer.append(mem, mem_size, false);

				sub_object->unpack(temp_buffer);
			}
		}

	protected:
		virtual std::shared_ptr<object_base> _create(const std::string& owner, const std::string& var) const override
		{
			return std::make_shared<object>(owner, var);
		}

	private:

#pragma pack(push, 1)
		struct pack_buffer_s
		{
			uint64_t offset;
			uint64_t size;
		};

		struct pack_header_s
		{
			uint64_t typename_ofs;
			uint64_t owner_ofs;
			uint64_t name_ofs;

			uint64_t num_vars;
			uint64_t num_children;

			struct pack_buffer_s strings;
			struct pack_buffer_s vars;
			struct pack_buffer_s children;
			struct pack_buffer_s data;
		};

		struct pack_var_s
		{
			uint64_t owner_ofs;
			uint64_t name_ofs;
			uint64_t type_ofs;

			uint64_t data_ofs;
			uint64_t bytes_size;

		};

		struct pack_child_s
		{
			uint64_t owner_ofs;
			uint64_t name_ofs;

			uint64_t data_ofs;
			uint64_t size;
		};
#pragma pack(pop)
	};
};
