#pragma once

#include "../../debug/safe.h"

#include "../../exception.h"

#include "../base.h"
#include "../fs/object.h"
#include "../exception.h"

#include <string>

namespace storage::ext
{
	// storeable version of std::string
	class string : public std::string, public storage::base
	{
	public:
		using std::string::string;
		using std::string::operator=;

		virtual std::string get_typename(void) const override
		{
			return "ext::string";
		}

		// push to storage object
		virtual void push(storage::fs::object_base& container) const
		{
			size_t bytesize = MAKE_ADD(length(), (size_t)1);

			// write size
			container.push_var(get_typename(), "size", "size_t", sizeof(bytesize), (void*)&bytesize);

			// write data
			container.push_var(get_typename(), "data", "char", MAKE_MULT(bytesize, sizeof(char)), (void*)c_str());

			// set typename
			container.set_type_name(get_typename());
		}

		// pop from storage object
		virtual void pop(const storage::fs::object_base& container)
		{
			// check typename
			if (container.get_type_name() != get_typename())
				throw_exception(wrong_type_exception);

			size_t bytesize = 0;

			// read size
			container.pop_var(get_typename(), "size", "size_t", sizeof(bytesize), (void*)&bytesize);

			// allocate space and read data
			char* string = (char*)malloc(bytesize);

			if (string == nullptr)
				throw_exception(memory_allocation_exception, bytesize);

			try
			{
				container.pop_var(get_typename(), "data", "char", MAKE_MULT(bytesize, sizeof(char)), string);

				this->operator=(std::string(string));
			}
			catch (...)
			{
				// free temporary string
				if (string != nullptr)
					free(string);

				string = nullptr;

				throw_exception(unknown_var_exception);
			}

			// free temporary string
			if (string != nullptr)
				free(string);

			string = nullptr;
		}
	};
};