#pragma once

#include "../../debug/safe.h"

#include "../../exception.h"

#include "../buffer.h"
#include "../base.h"
#include "../fs/object.h"
#include "../exception.h"

#include <string>
#include <vector>
#include <array>

namespace storage::ext
{
	// storeable version of std::string
	class buffer : public storage::buffer, public storage::base
	{
	public:
		using storage::buffer::buffer;
		using storage::buffer::operator=;

		virtual std::string get_typename(void) const override
		{
			return "ext::buffer";
		}

		// push to storage object
		virtual void push(storage::fs::object_base& obj) const
		{
			auto bytesize = size();

			// write size
			obj.push_var(get_typename(), "size", "size_t", sizeof(bytesize), (void*)&bytesize);

			// write data
			obj.push_var(get_typename(), "data", "unsigned char", MAKE_MULT(bytesize, sizeof(unsigned char)), (void*)data());

			// set typename
			obj.set_type_name(get_typename());
		}

		// pop from storage object
		virtual void pop(const storage::fs::object_base& obj)
		{
			// check typename
			if (obj.get_type_name() != get_typename())
				throw_exception(wrong_type_exception);

			// free existing data
			clear();

			// read size
			size_t bytesize;

			try
			{

				obj.pop_var(get_typename(), "size", "size_t", sizeof(bytesize), (void*)&bytesize);
			}
			catch (...)
			{
				throw_exception(unknown_var_exception);
			}

			// allocate space and read data
			auto data = (unsigned char*)malloc(bytesize);

			if (data == nullptr)
				throw_exception(memory_allocation_exception, bytesize);

			try
			{
				obj.pop_var(get_typename(), "data", "unsigned char", MAKE_MULT(bytesize, sizeof(unsigned char)), data);
			}
			catch (...)
			{
				// free temporary data
				if (data != nullptr)
					free(data);

				throw_exception(unknown_var_exception);
			}

			// assign result
			this->operator=(storage::buffer(data, bytesize));
		}
	};
};