#include "../storage/storage.h"
#include "../utils/base64.h"
#include "../debug/evemon.h"

#include "uid.h"

INITIALIZE_SINGLETON_EX(resources::uid::manager, resources_uid_manager);

#ifdef WIN32

#include <WinSock2.h>
#include <WS2tcpip.h>
#include <iphlpapi.h>
#include <windows.h>

#pragma comment(lib, "Ws2_32.lib")
#pragma comment(lib, "IPHLPAPI.lib")

// create machine-based identifier
storage::buffer resources::uid::ident(void)
{
	storage::buffer ident;

	// get computer name
	{
		char buffer[MAX_COMPUTERNAME_LENGTH + 1];
		DWORD size = sizeof(buffer);

		if(GetComputerNameA(buffer, &size))
			ident.append((const unsigned char*)buffer, strlen(buffer));
	}

	// add volumes information
	{
		char buffer[MAX_PATH];
		memset(buffer, 0, sizeof(buffer));

		auto handle = FindFirstVolumeA(buffer, sizeof(buffer));

		if (handle != INVALID_HANDLE_VALUE)
		{
			do
			{
				ident.append((const unsigned char*)buffer, strlen(buffer));

				DWORD sn = 0;

				if (GetVolumeInformationA(buffer, NULL, 0, &sn, NULL, NULL, NULL, 0))
					ident.append<DWORD>(sn);

			} while (FindNextVolumeA(handle, buffer, sizeof(buffer)));

			FindVolumeClose(handle);
		}
	}

	// add ethernet adapters
	{
		DWORD dwSize = 0;

		GetAdaptersAddresses(AF_INET, GAA_FLAG_INCLUDE_PREFIX, NULL, NULL, &dwSize);

		PIP_ADAPTER_ADDRESSES pAdapters = (PIP_ADAPTER_ADDRESSES)malloc(dwSize);

		try
		{
			if (GetAdaptersAddresses(AF_INET, GAA_FLAG_INCLUDE_PREFIX, NULL, pAdapters, &dwSize) == ERROR_SUCCESS)
			{
				for (auto pCurr = pAdapters; pCurr != nullptr; pCurr = pCurr->Next)
				{
					if (pCurr == nullptr)
						continue;

					ident.append((const unsigned char*)pCurr->AdapterName, strlen(pCurr->AdapterName));
					ident.append((const unsigned char*)pCurr->PhysicalAddress, sizeof(pCurr->PhysicalAddress));

				}
			}
		}
		catch (...)
		{
			if (pAdapters != nullptr)
				free(pAdapters);

			pAdapters = nullptr;

			throw;
		}

		if (pAdapters != nullptr)
			free(pAdapters);

		pAdapters = nullptr;
	}

	// return identification buffer
	_debug("resolving machine identification as \"%s\"", utils::base64::encode(ident).c_str());

	return ident;
}

#else

#include <unistd.h>
#include <ifaddrs.h>
#include <netpacket/packet.h>

// create machine-based identifier
storage::buffer resources::uid::ident(void)
{
	storage::buffer ident;

	// get computer name
	{
		char name[HOST_NAME_MAX];

		gethostname(name, sizeof(name));

		ident.append((unsigned char*)name, strlen(name), false);
	}

	// add volumes information
	{
		// TODO
	}

	// add ethernet adapters
	{
		struct ifaddrs *ifa = nullptr;

		if (getifaddrs(&ifa) == 0)
		{
			for (struct ifaddrs* p = ifa; p != nullptr; p = p->ifa_next)
				if (p->ifa_addr && p->ifa_addr->sa_family == AF_PACKET)
				{
					auto s = (struct sockaddr_ll*)ifa->ifa_addr;

					ident.append((const unsigned char*)s->sll_addr, sizeof(s->sll_addr), false);
				}

			freeifaddrs(ifa);
		}
	}

	// return identification buffer
	_debug("resolving machine identification as \"%s\"", utils::base64::encode(ident).c_str());

	return ident;
}


#endif
