#pragma once

#include "../../utils/utils.h"
#include "../../debug/exception.h"
#include "../../debug/evemon.h"
#include "../../debug/exdebug.h"
#include "../../types/type.h"
#include "../../storage/file.h"
#include "../../storage/storage.h"

#include "exception.h"
#include "state.h"
#include "base.h"

#include <memory>
#include <string>
#include <mutex>
#include <unordered_map>
#include <filesystem>

namespace net::session
{
	template<class msg_type> class db_base
	{
	public:
		using session_t = session_base<msg_type>;

	public:
		virtual std::shared_ptr<session_t> create(net::endpoint<msg_type>& endpoint, const std::string& id) = 0;
		virtual bool has(const std::string& id) const = 0;
		virtual std::shared_ptr<session_t> get(const std::string& id) = 0;
		virtual void offline(const std::string& id) = 0;
	};

	/* base class */
	template<class session_type> class db : public db_base<typename session_type::msg>
	{
	public:
		using msg_type = typename session_type::msg;
		using session_t = session_base<msg_type>;

		// default ctor, no saving
		db(void) {}

		// ctor
		db(const std::string& directory)
		{
			_debug("outputting sessions to %s...", directory.c_str());

			this->m_directory = std::make_unique<std::string>(directory);

			// check if directory exists
			switch (std::filesystem::status(directory).type())
			{
				// already existing, do nothing
			case std::filesystem::file_type::directory:
				break;

				// not found, create directory
			case std::filesystem::file_type::not_found:
				std::filesystem::create_directory(directory);
				break;

				// all other cases should trigger an error
			default:
				throw_exception(cannot_create_directory_exception, directory);
			}
		}

		// create session
		std::shared_ptr<session_t> create(net::endpoint<msg_type>& endpoint, const std::string& id)
		{
			// throw exception if already existing
			if (has(id))
				throw_exception(session_already_exist_exception, id);

			// create session
			_debug("creating session %s...", id.c_str());

			auto p = std::make_shared<session_type>(id);

			if (p == nullptr)
				return nullptr;

			// add to list
			{
				AUTOLOCK(this->m_lock);

				this->m_sessions.emplace(std::make_pair(id, p));
			}

			// return pointer
			return p;
		}

		// return true if session id exist
		bool has(const std::string& id) const
		{
			// check first in opened sessions
			{
				AUTOLOCK(this->m_lock);

				if (this->m_sessions.find(id) != this->m_sessions.end())
					return true;
			}

			// now check on HDD
			if (this->m_directory != nullptr)
			{
				auto filename = id2filename(id);

				if (std::filesystem::exists(filename))
					return true;
			}

			// session does not exist
			return false;
		}

		// return session
		std::shared_ptr<session_t> get(const std::string& id)
		{
			// check in opened session first
			{
				AUTOLOCK(this->m_lock);

				auto it = this->m_sessions.find(id);

				if (it != this->m_sessions.end())
					return it->second;
			}

			// check on HDD
			if (this->m_directory != nullptr)
			{
				auto filename = id2filename(id);

				if (std::filesystem::exists(filename))
				{
					_debug("restoring session %s...", id.c_str());

					storage::container container;

					// MUST be session type
					auto _create = [=](const std::string& name)
					{
						if (!is_type_name<session_type>(name))
							throw_exception(wrong_type_exception);

						return std::make_shared<session_type>(id);
					};

					// unpack data from file
					container.unpack(storage::load_buffer_from_file(filename), _create);

					// can have only one object
					if (container.size() != 1)
						throw_exception(failed_loading_session_exception, filename);

					// retrieve object and add to open list
					auto [owner, var, obj_ptr] = container.front();

					auto p = std::dynamic_pointer_cast<session_t, storage::base>(obj_ptr);

					{
						AUTOLOCK(this->m_lock);

						this->m_sessions.emplace(std::make_pair(id, p));
					}

					return p;
				}
			}

			// otherelse throw exception
			throw_exception(unknown_session_exception, id);
		}

		// save given id to disk
		virtual void offline(const std::string& id) override
		{
			_debug("offlining session %s...", id.c_str());

			std::shared_ptr<session_t> p;

			if (this->m_directory == nullptr)
				throw_exception(cannot_offline_exception);

			{
				AUTOLOCK(this->m_lock);

				// retrieve session from opened sessions
				auto it = this->m_sessions.find(id);

				if (it == this->m_sessions.end())
					throw_exception(unknown_session_exception, id);

				p = it->second;
			}

			// pack to storage file
			storage::container container;

			container.push_back(p);

			auto buffer = container.pack('SPC0');

			// save to file
			auto filename = id2filename(id);

			storage::save_buffer_to_file(filename, buffer);

			// remove from open list
			{
				AUTOLOCK(this->m_lock);

				auto it = this->m_sessions.find(id);

				if (it != this->m_sessions.end())
					this->m_sessions.erase(it);
			}

		}

		// save all sessions to disk
		void offline_all(void)
		{
			for (auto& v : this->m_sessions)
				offline(v.first);
		}

	private:

		std::string id2filename(const std::string& id) const
		{
			if (this->m_directory == nullptr)
				throw_exception(cannot_offline_exception);

			return build_path({ *this->m_directory, storage::buf2hex(sha256(id)) });
		}

	private:
		mutable std::mutex m_lock;
		std::unordered_map<std::string, std::shared_ptr<session_t>> m_sessions;
		std::unique_ptr<std::string> m_directory;
	};
};