#pragma once

#include "../../utils/utils.h"
#include "../../debug/exception.h"
#include "../../debug/evemon.h"
#include "../../debug/exdebug.h"
#include "../../instancers/local_instancer.h"
#include "../../storage/base.h"

#include "exception.h"
#include "state.h"
#include "msg.h"

#include <string>
#include <memory>
#include <mutex>
#include <list>

namespace net::session
{
	/* fsm class */
	template<class msg_type> class session_base : public storage::base, public instancer::local<state<msg_type>, session_base<msg_type>&>
	{
	public:
		IMPLEMENT_STORAGE;

		using state_t = state<msg_type>;
		using msg = msg_type;

		friend class state<msg_type>;

	public:

		// ctor
		session_base(const std::string& id)
		{
			this->m_id = id;
		}

		// change current state
		void push_state(std::shared_ptr<state_t> p)
		{
			if (__ISNULLPTR(p))
				return;

			_debug("pushing state %s to session %s", p->get_typename().c_str(), this->m_id.c_str());

			AUTOLOCK(this->m_lock);

			this->m_stack.push_back(p);
		}

		// retrieve previous state
		bool pop_state(void)
		{
			AUTOLOCK(this->m_lock);

			if (this->m_stack.empty())
				return false;

			this->m_stack.pop_back();
		}

		// get current state
		std::shared_ptr<state_t> get_state(void)
		{
			AUTOLOCK(this->m_lock);

			if (this->m_stack.empty())
				return nullptr;

			return this->m_stack.back();
		}

		// handle message
		bool handle(const storage::buffer& msg, net::endpoint<msg_type>& endpoint)
		{
			auto p = get_state();

			if (p == nullptr)
				return false;

			return p->process(msg, endpoint);
		}

	public:
		virtual std::shared_ptr<state_t> create_default_state(void) = 0;

	private:

		// send message
		bool send(const msg_type& msg, net::endpoint<msg_type>& endpoint)
		{
			auto data = endpoint.pack(msg);

			return endpoint.send(net::session::msg_base<msg_type>(this->m_id, data));
		}

		// push all states
		virtual void custom_push(storage::fs::object_base& obj) const override
		{
			AUTOLOCK(this->m_lock);

			// browse all elements
			for (auto& p : this->m_stack)
			{
				if (p == nullptr)
					continue;

				// add as sub objects
				auto sub_obj = obj.create_sub_object("", "stack");

				if (sub_obj == nullptr)
					continue;

				p->push(*sub_obj);
			}
		}

		// pop all states
		virtual void custom_pop(const storage::fs::object_base& obj) override
		{
			// clear existing data
			{
				AUTOLOCK(this->m_lock);

				this->m_stack.clear();
			}

			// browse all sub objects
			auto children = obj.get_children();

			for (auto& p : children)
			{
				if (p == nullptr)
					continue;

				// check that it is stack
				if (p->get_var_name() != "stack")
					continue;

				// create object by typename
				auto obj = this->create_instance(p->get_type_name(), *this);

				if (obj == nullptr)
					continue;

				// import data
				obj->pop(*p);

				// add to list
				{
					AUTOLOCK(this->m_lock);

					this->m_stack.push_back(obj);
				}
			}
		}

	private:
		std::string m_id;
		mutable std::mutex m_lock;
		std::list<std::shared_ptr<state_t>> m_stack;
	};
};

#define REGISTER_SESSION_MESSAGE_EX(class, type, func)						register_listener<type>(std::bind(&class::func, this, std::placeholders::_1, std::placeholders::_2));
#define REGISTER_SESSION_MESSAGE(class, type)								REGISTER_SESSION_MESSAGE_EX(class, type, recv_##type);

#define HANDLE_SESSION_MESSAGE_EX(type, obj, endpoint_obj, msg_type, func)	void func(const type& obj, net::endpoint<msg_type>& endpoint_obj)
#define HANDLE_SESSION_MESSAGE(type, obj, endpoint, msg_type)				HANDLE_SESSION_MESSAGE_EX(type, obj, endpoint, msg_type, recv_##type)

#define REGISTER_SESSION_MESSAGE_OBJECT(class, type)						DECLARE_LOCAL_INSTANCER(type); \
																			REGISTER_SESSION_MESSAGE(class, type)

// declare instanceable object for given class/type
#define DECLARE_SESSION_STATE_OBJECT_EX(class, name, type, msg_type)		std::shared_ptr<type> create_instance__##name(net::endpoint<msg_type> &endpoint)			\
																			{																							\
																				return type :: create<class>(endpoint);													\
																			}																							\
																																									\
																			REGISTER_TYPENAME_EX(class, #class, name);													\
																																									\
																			instancer::global< type > :: auto_register g_register_instancer__##name(::get_name_from_type< class >(), create_instance__##name);

#define DECLARE_SESSION_STATE_OBJECT(class, type, msg_type)					DECLARE_SESSION_STATE_OBJECT_EX(class, class, type, msg_type)