#pragma once

#include "../hash/sha2.h"
#include "../hash/checksum32.h"
#include "../debug/safe.h"
#include "../types/endian.h"
#include "../storage/buffer.h"
#include "../event/notify.h"

#include "exception.h"

namespace net
{
	/* packet class */
	class packet : public event::notify::base<>
	{
	public:
		enum
		{
			EVENT_WRONG_PACKET_CHECKSUM=0,
			EVENT_WRONG_MESSAGE_CHECKSUM,
			EVENT_INVALID_IDENT,
		};

		// initialize packet
		packet(const uint32_t ident, const size_t max_size=-1)
		{
			auto i = htof(ident);

			this->m_ident = sha256((unsigned char*)&i, sizeof(i));
			this->m_maxsize = max_size;
		}

		// set maximum packet size
		void set_maxsize(const size_t max_size)
		{
			this->m_maxsize = max_size;
		}

		// pack bytes into a packet
		storage::buffer pack(const unsigned char* bytes, const size_t size) const
		{
			// allocate buffer
			storage::buffer buffer;
			
			buffer.allocate(MAKE_ADD(sizeof(struct header_s), size));

			// copy bytes data
			memcpy(buffer.ptr<unsigned char>(sizeof(struct header_s), size), bytes, size);

			// set header
			auto hdr = buffer.ptr<struct header_s>(0, sizeof(struct header_s));

			memcpy(hdr->ident, this->m_ident.data(), std::min(sizeof(hdr->ident), this->m_ident.size()));

			hdr->size = MAKE_UINT64(htof(buffer.size()));
			hdr->pksize = MAKE_UINT64(htof(size));
			hdr->pkcksum = htof(checksum32(bytes, size));
			
			hdr->cksum = 0;
			hdr->cksum = htof(checksum32((const unsigned char*)hdr, sizeof(struct header_s)));

			// return packet
			return buffer;
		}

		// pack storage buffer
		storage::buffer pack(const storage::buffer& buf) const
		{
			return pack(buf.data(), buf.size());
		}

		// unpack a packet
		void unpack(const unsigned char* bytes, const size_t size)
		{
			// trow exception if size is too large
			if (MAKE_ADD(this->m_curr.size(), size) >= this->m_maxsize)
				throw_exception(packet_too_large_exception);

			// append to current message
			this->m_curr.append(bytes, size, false);

			// try processing
			try_process();
		}

	protected:
		virtual void on_process(const unsigned char* bytes, const size_t size) = 0;

	private:
		// try to process data
		void try_process(void)
		{
			size_t ofs = 0;

			// loop as long as there are bytes
			while (this->m_curr.size() > MAKE_ADD(sizeof(struct header_s), ofs))
			{
				// get header copy
				auto hdr = *this->m_curr.ptr<struct header_s>(ofs, sizeof(struct header_s));

				// check ident, go to next offset if not matching
				if (memcmp(hdr.ident, this->m_ident.data(), std::min(sizeof(hdr.ident), this->m_ident.size())) != 0)
				{
					ofs = MAKE_ADD(ofs, (size_t)1);

					notify(EVENT_INVALID_IDENT);

					continue;
				}

				// verify header checksum
				auto packet_cksum = ftoh(hdr.cksum);
				hdr.cksum = 0;

				// skip bytes if checksum does not match
				if (checksum32((const unsigned char*)&hdr, sizeof(struct header_s)) != packet_cksum)
				{
					ofs = MAKE_ADD(ofs, sizeof(struct header_s));

					notify(EVENT_WRONG_MESSAGE_CHECKSUM);

					continue;
				}

				// check if enough bytes, abort if not
				auto size = MAKE_SIZE_T(ftoh(hdr.size));
				auto pksize = MAKE_SIZE_T(ftoh(hdr.pksize));

				if (this->m_curr.size() < MAKE_ADD(size, ofs))
					break;

				// get packet
				auto pkdata = this->m_curr.ptr<const unsigned char>(MAKE_ADD(ofs, sizeof(struct header_s)), pksize);

				// check packet checksum
				auto pkcksum = ftoh(hdr.pkcksum);

				if (checksum32(pkdata, pksize) != pkcksum)
				{
					ofs = MAKE_ADD(ofs, pksize);

					notify(EVENT_WRONG_PACKET_CHECKSUM);

					continue;
				}

				// process message
				on_process(pkdata, pksize);

				// goto next
				ofs = MAKE_ADD(ofs, pksize);
			}

			// clear nOffset bytes
			if (ofs != 0)
				this->m_curr = storage::buffer(MAKE_ADDRESS(this->m_curr.data(), ofs), MAKE_SUB(this->m_curr.size(), ofs));
		}

		struct header_s
		{
			uint8_t ident[32];
			uint64_t size;
			uint32_t cksum;
			uint64_t pksize;
			uint32_t pkcksum;
		};

		size_t m_maxsize;

		storage::buffer m_ident;
		storage::buffer m_curr;
	};
}
