#pragma once

#include "../message/handler.h"
#include "../scope/access.h"

#include "exception.h"
#include "manager.h"

#include <string>
#include <memory>

namespace message
{
	/* entity goc class */
	template<class msg_base> class entity : public handler<msg_base, std::string>, public std::enable_shared_from_this<entity<msg_base>>
	{
	public:
		using token_t = typename manager<msg_base>::token;

		// return uid
		auto uid(void) const
		{
			return self().uid();
		}

		// return name
		auto name(void) const
		{
			return self().name();
		}

		// send message
		bool send(const std::string& dest, const msg_base& msg) const
		{
			return self().dispatch(dest, msg);
		}

		// broadcast message
		bool send(const std::list<std::string>& dests, const msg_base& msg) const
		{
			bool ret = false;

			for(auto& v : dests)
				ret |= self().dispatch(v, msg);

			return ret;
		}

		// set name
		void set_name(const std::string& name)
		{
			self().set_name(name);
		}

		// get list of all other entities
		std::list<std::string> all(const std::list<std::string>& excludes = {}) const
		{
			return get_instance<manager<msg_base>>()->all(excludes);
		}

		// inverse resolve
		auto inv_resolve(const std::string& uid) const
		{
			return get_instance<manager<msg_base>>()->inv_resolve(uid);
		}

	protected:

		// return self uid
		token_t& self(void) const
		{
			if (this->m_token == nullptr)
				const_cast<entity*>(this)->register_self();

			if (this->m_token == nullptr)
				throw_exception(cannot_identify_self_exception);

			return *this->m_token;
		}

	private:

		// register to manager
		void register_self(void)
		{
			this->m_token = get_instance<manager<msg_base>>()->register_entity(this->shared_from_this());
		}

		// members
		std::unique_ptr<token_t> m_token;
	};
};

#define REGISTER_ENTITY_MESSAGE_EX(class, type, func)				register_listener<type>(std::bind(&class::func, this, std::placeholders::_1, std::placeholders::_2));
#define REGISTER_ENTITY_MESSAGE(class, type)						REGISTER_ENTITY_MESSAGE_EX(class, type, recv_##type);

#define HANDLE_ENTITY_MESSAGE_EX(type, obj, sender_id, func)		void func(const type& obj, const std::string& sender_id)
#define HANDLE_ENTITY_MESSAGE(type, obj, sender_id)					HANDLE_ENTITY_MESSAGE_EX(type, obj, sender_id, recv_##type)