#pragma once

#include "../utils/utils.h"
#include "../debug/exception.h"
#include "../debug/safe.h"

#include "mem_manager.h"
#include "exception.h"

/* bunch of data type */
template<typename type> class memory
{
public:

	// default constructor
	memory(void)
	{
		this->m_num_elems = 0;
		this->m_data = nullptr;
		this->m_memory_manager = nullptr;
	}

	// allocate data
	memory(size_t num_elems) : memory()
	{
		realloc(num_elems);
	}

	// allocate and copy data
	memory(type* p, size_t num_elems) : memory()
	{
		realloc(num_elems);
		copy_from(p, num_elems);
	}

	// copy constructor
	memory(const memory<type>& obj) : memory()
	{
		this->operator=(obj);
	}

	// move destructor
	memory(memory<type>&& obj) : memory()
	{
		this->operator=(std::move(obj));
	}

	// destructor
	~memory(void)
	{
		dealloc();
	}

	// copy operator
	const memory& operator=(const memory<type>& obj)
	{
		// reallocate memory unless same size
		if (obj.num_elems() != num_elems())
			realloc(obj.num_elems());

		// fill in
		copy_from(obj.m_data, num_elems());

		return *this;
	}

	// move operator
	const memory& operator=(memory<type>&& obj)
	{
		// reassign pointer
		this->m_data = obj.m_data;
		this->m_memory_manager = obj.m_memory_manager;
		this->m_num_elems = obj.m_num_elems;

		// clear old pointers
		obj.m_data = nullptr;
		obj.m_memory_manager = nullptr;
		obj.m_num_elems = 0;

		return *this;
	}

	// clear
	void clear(void)
	{
		dealloc();
	}

	// return true if valid
	bool is_valid(void) const
	{
		return this->m_data != nullptr;
	}

	// return number of elements
	size_t num_elems(void) const
	{
		return this->m_num_elems;
	}

	// return size
	size_t size(void) const
	{
		return MAKE_MULT(this->m_num_elems, sizeof(type));
	}

	// get const pointer
	const type* ptr(void) const
	{
		return this->m_data;
	}

	// copy from data
	void copy_from(const type* p, size_t num_elems, size_t offset = 0)
	{
		// check if enough buffer size
		if (MAKE_ADD(num_elems, offset) > this->m_num_elems)
			throw_exception(memory_out_of_bound_exception);

		// perform copy
		memcpy(MAKE_AT(this->m_data, offset), p, MAKE_MULT(num_elems, sizeof(type)));
	}

	// copy to external buffer
	void copy_to(type* p, size_t num_elems, size_t offset = 0)
	{
		// check if enough buffer size
		if (MAKE_ADD(num_elems, offset) > this->m_num_elems)
			throw_exception(memory_out_of_bound_exception);

		// perform copy
		memcpy(p, MAKE_AT(this->m_data, offset), MAKE_MULT(num_elems, sizeof(type)));
	}

	// fill in data
	void fill(const type& data, size_t num_elems, size_t offset = 0)
	{
		// check if enough buffer size
		if (MAKE_ADD(num_elems, offset) > this->m_num_elems)
			throw_exception(memory_out_of_bound_exception);

		// fill in data
		auto p = MAKE_AT(this->m_data, offset);

		while (num_elems--)
			*(p++) = data;
	}

	// get data at
	type& at(const size_t index)
	{
		if(index >= this->m_num_elems)
			throw_exception(memory_out_of_bound_exception);

		return *MAKE_AT(this->m_data, index);
	}

	// get data at (const version)
	const type& at(const size_t index) const
	{
		if (index >= this->m_num_elems)
			throw_exception(memory_out_of_bound_exception);

		return *MAKE_AT(this->m_data, index);
	}

private:

	// allocate memory
	void realloc(size_t num_elems)
	{
		// always dealloc first
		dealloc();

		// acquire page
		this->m_memory_manager = get_instance<memory_manager>();
		this->m_data = static_cast<type*>(this->m_memory_manager->acquire(MAKE_MULT(num_elems, sizeof(type))));

		// set number of elements
		this->m_num_elems = num_elems;
	}

	// deallocate memory
	void dealloc(void)
	{
		// release page if existing
		if(this->m_memory_manager != nullptr && this->m_data != nullptr)
			this->m_memory_manager->release(this->m_data);

		// clear fields
		this->m_memory_manager = nullptr;
		this->m_data = nullptr;
		this->m_num_elems = 0;
	}

	// member variables
	memory_manager* m_memory_manager;
	type* m_data;
	size_t m_num_elems;
};
