#pragma once

#include "../utils/utils.h"
#include "../utils/singleton.h"
#include "../debug/exception.h"
#include "../debug/exdebug.h"
#include "../debug/evemon.h"
#include "../strings/fileparts.h"

#include "base.h"
#include "so.h"
#include "own.h"
#include "exception.h"

#include <string>
#include <memory>
#include <unordered_map>

namespace library
{
	/* manager */
	class manager : public singleton<manager>
	{
		friend class singleton<manager>;

	public:

		// detach all libraries on dtor
		~manager(void)
		{
			detach_all();
		}

		// set library name
		void set_name(const std::string& name)
		{
			if (name.find('/') != std::string::npos)
				throw_exception(invalid_libname_exception, name);

			this->m_name = name;

			_debug("library name changed to \"%s\"", name.c_str());
		}

		// get library name
		const std::string& get_name(void) const
		{
			return this->m_name;
		}

		// detach all libraries
		void detach_all(void);

		// return true if library exists
		bool has(const std::string& filename) const
		{
			return this->m_libraries.find(filename) != this->m_libraries.end();
		}

		// load given file
		void load(const std::string& filename)
		{
			// load library
			auto p = std::make_shared<so>(filename);

			// get name
			auto func = (const char*(*)(void))p->get_addr("get_library_name");

			std::string libname((*func)());

			_debug("%s resolved as %s", filename.c_str(), libname.c_str());

			// throw error if it already exists
			if (has(libname))
				throw_exception(library_already_exists_exception, libname);

			// add to objects
			this->m_libraries.emplace(std::make_pair(libname, p));
		}

		// get func address from given library
		void* get_addr(const std::string& library, const std::string& funcname)
		{
			// get library
			auto it = this->m_libraries.find(library);

			if (it == this->m_libraries.end())
				throw_exception(unknown_library_exception, library);

			// sanity check
			if (__ISNULLPTR(it->second))
				throw_exception(invalid_library_exception, library);

			// return function pointer
			return it->second->get_addr(funcname);
		}

	private:

		// add self on ctor
		manager(void)
		{
			_debug("initializing library manager...");

			this->m_libraries.emplace(std::make_pair(std::string(""), std::make_shared<own>()));
		}

	private:
		std::unordered_map<std::string, std::shared_ptr<base>> m_libraries;
		std::string m_name;
	};
};