#pragma once

#include "../utils/utils.h"
#include "../debug/safe.h"

#include "map2d.h"

#include <array>

template<typename type> struct grid_s
{
	map2d<type> xx, yy;
};

template<typename type> struct grid_s<type> make_grid(const map2d<type>& xx, const map2d<type>& yy)
{
	struct grid_s<type> ret;

	ret.xx = xx;
	ret.yy = yy;

	return ret;
}

template<typename type> struct grid_s<type> make_grid(map2d<type>&& xx, const map2d<type>& yy)
{
	struct grid_s<type> ret;

	ret.xx = std::move(xx);
	ret.yy = yy;

	return ret;
}

template<typename type> struct grid_s<type> make_grid(const map2d<type>& xx, map2d<type>&& yy)
{
	struct grid_s<type> ret;

	ret.xx = xx;
	ret.yy = std::move(yy);

	return ret;
}

template<typename type> struct grid_s<type> make_grid(map2d<type>&& xx, map2d<type>&& yy)
{
	struct grid_s<type> ret;

	ret.xx = std::move(xx);
	ret.yy = std::move(yy);

	return ret;
}

template<typename type> struct grid_s<type> meshgrid(const size_t width, const size_t height)
{
	throw_exception(template_func_not_impl_exception, typeid(type).name());
}

template<> struct grid_s<float> meshgrid<float>(const size_t width, const size_t height)
{
	struct grid_s<float> ret;

	ret.xx = map2d<float>(width, height);

	ret.xx.perpixel([](size_t x, size_t y)
		{
			return (float)x;
		});

	ret.yy = map2d<float>(width, height);

	ret.yy.perpixel([](size_t x, size_t y)
		{
			return (float)y;
		});

	return ret;
}

template<> struct grid_s<double> meshgrid<double>(const size_t width, const size_t height)
{
	struct grid_s<double> ret;

	ret.xx = map2d<double>(width, height);

	ret.xx.perpixel([](size_t x, size_t y)
		{
			return (double)x;
		});

	ret.yy = map2d<double>(width, height);

	ret.yy.perpixel([](size_t x, size_t y)
		{
			return (double)y;
		});

	return ret;
}

template<> struct grid_s<long double> meshgrid<long double>(const size_t width, const size_t height)
{
	struct grid_s<long double> ret;

	ret.xx = map2d<long double>(width, height);

	ret.xx.perpixel([](size_t x, size_t y)
		{
			return (long double)x;
		});

	ret.yy = map2d<long double>(width, height);

	ret.yy.perpixel([](size_t x, size_t y)
		{
			return (long double)y;
		});

	return ret;
}

template<typename type_grid, typename type_map> struct grid_s<type_grid> meshgrid(const map2d<type_map>& obj)
{
	return meshgrid<type_grid>(obj.width(), obj.height());
}

template<typename type> struct grid_s<type> create_centered_grid(size_t width, size_t height, const std::array<type, 2> pixelsizes, const std::array<type, 2> offsets)
{
	throw_exception(template_func_not_impl_exception, typeid(type).name());
}

template<> struct grid_s<float> create_centered_grid(size_t width, size_t height, const std::array<float, 2> pixelsizes, const std::array<float, 2> offsets)
{
	auto grid = meshgrid<float>(width, height);

	grid.xx = pixelsizes[0] * grid.xx + (offsets[0] - (pixelsizes[0] * (0.5f * (float)MAKE_SUB(width, (size_t)1))));
	grid.yy = pixelsizes[1] * grid.yy + (offsets[1] - (pixelsizes[1] * (0.5f * (float)MAKE_SUB(height, (size_t)1))));

	return grid;
}

template<typename type> struct grid_s<type> create_centered_grid(const map2d<type>& obj, const std::array<type, 2> pixelsizes, const std::array<type, 2> offsets)
{
	return create_centered_grid(obj.width(), obj.height(), pixelsizes, offsets);
}
