#pragma once

#include "../utils/utils.h"
#include "../debug/exception.h"
#include "../debug/safe.h"
#include "../memory/memory.h"

#include "exception.h"

#include <list>
#include <fstream>
#include <functional>

/* template class for 2D arrays of any types */
template<typename type> class map2d
{
public:

	// default constructor
	map2d(void)
	{
		this->m_width = 0;
		this->m_height = 0;
	}

	// allocate width x height elements
	map2d(size_t width, size_t height)
	{
		this->m_width = width;
		this->m_height = height;

		this->m_data = memory<type>(MAKE_MULT(this->m_width, this->m_height));
	}

	// copy constructor
	map2d(const map2d& obj)
	{
		this->m_width = 0;
		this->m_height = 0;

		this->operator=(obj);
	}

	// move operator
	map2d(map2d&& obj) noexcept
	{
		this->m_width = 0;
		this->m_height = 0;

		this->operator=(std::move(obj));
	}

	// destructor
	~map2d(void)
	{
		clear();
	}

	// copy operator
	const map2d<type>& operator=(const map2d<type>& obj)
	{
		// delete previous data if any
		clear();

		// allocate size
		this->m_width = obj.m_width;
		this->m_height = obj.m_height;

		// copy data
		this->m_data = obj.m_data;

		return *this;
	}

	// move operator
	const map2d<type>& operator=(map2d<type>&& obj) noexcept
	{
		// delete previous data if any
		clear();

		// move things
		this->m_width = obj.m_width;
		this->m_height = obj.m_height;
		this->m_data = std::move(obj.m_data);

		// clear moved object
		obj.m_width = 0;
		obj.m_height = 0;

		return *this;
	}

	// clear map
	void clear(void)
	{
		this->m_data.clear();

		this->m_width = 0;
		this->m_height = 0;
	}

	// check if valid
	bool is_valid(void) const
	{
		return this->m_data.is_valid();
	}

	// return width
	const size_t width(void) const
	{
		return this->m_width;
	}

	// return height
	const size_t height(void) const
	{
		return this->m_height;
	}

	// fill map with a single value
	const map2d<type>& operator=(const type value)
	{
		this->m_data.fill(value, this->m_data.num_elems());

		return *this;
	}

	// apply function for each lines
	void perline(std::function<void(size_t)> func, size_t margin_y = 0)
	{
		// skip if no function
		if (!func)
			return;

		// skip if size if below margin
		if (this->m_height <= margin_y)
			return;

		// browse all points
		for (size_t y = margin_y; y < MAKE_SUB(this->m_height, margin_y); y++)
			func(y);
	}

	// apply function for each pixel
	void perpixel(std::function<type(size_t, size_t)> func, size_t margin_x=0, size_t margin_y=0)
	{
		// skip if no function
		if (!func)
			return;

		// skip if size if below margin
		if (this->m_width <= margin_x || this->m_height <= margin_y)
			return;

		// browse all points
		for (size_t y = margin_y; y < MAKE_SUB(this->m_height, margin_y); y++)
			for (size_t x = margin_x; x < MAKE_SUB(this->m_width, margin_x); x++)
				this->m_data.at(MAKE_XY(x, y, this->m_width)) = func(x, y);
	}

	// get pixel (non-const version)
	type& operator()(size_t x, size_t y)
	{
		// throw error if beyond dimensions
		if (x >= this->m_width || y >= this->m_height)
			throw_exception(array_out_of_bounds_exception);

		// return pixel reference
		return this->m_data.at(MAKE_XY(x, y, this->m_width));
	}

	// get pixel (const version)
	const type operator()(size_t x, size_t y) const
	{
		// throw error if beyond dimensions
		if (x >= this->m_width || y >= this->m_height)
			throw_exception(array_out_of_bounds_exception);

		// return pixel data
		return this->m_data.at(MAKE_XY(x, y, this->m_width));
	}

private:
	size_t m_width, m_height;

	memory<type> m_data;
};

// return width
template<typename type> size_t width(const map2d<type>& obj)
{
	return obj.width();
}

// return height
template<typename type> size_t height(const map2d<type>& obj)
{
	return obj.height();
}

// XY func
template<typename type_ret> map2d<type_ret> map2d_xyfunc(const size_t width, const size_t height, std::function<type_ret(size_t, size_t)> func)
{
	map2d<type_ret> ret(width, height);

	if (!func)
		throw_exception(map_function_exception);

	for (size_t y = 0; y < height; y++)
		for (size_t x = 0; x < width; x++)
			ret(x, y) = func(x, y);

	return ret;
};

// pixel-to-pixel conversion (1 op)
template<typename type_in, typename type_ret> map2d<type_ret> map2d_in2out(const map2d<type_in>& in, std::function<type_ret(type_in)> func)
{
	map2d<type_ret> ret(in.width(), in.height());

	if (!func)
		throw_exception(map_function_exception);

	for (size_t y = 0; y < in.height(); y++)
		for (size_t x = 0; x < in.width(); x++)
			ret(x, y) = func(in(x, y));

	return ret;
};

// reduction function (1 op)
template<typename type_in, typename type_kernel, typename type_ret> type_ret map2d_redux(const map2d<type_in>& obj, std::function<type_kernel(const type_kernel&, type_in)> kernel_func, std::function<type_ret(const std::list<type_kernel>&)> redux_func, std::shared_ptr<const map2d<bool>> mask = nullptr)
{
	if (mask != nullptr && (mask->width() != obj.width() || mask->height() != obj.height()))
		throw_exception(map_size_exception, obj.width(), obj.height(), mask->width(), mask->height());

	type_kernel tmp;

	for (size_t y = 0; y < obj.height(); y++)
		for (size_t x = 0; x < obj.width(); x++)
		{
			if (mask != nullptr && !(*mask)(x, y))
				continue;

			tmp = kernel_func(tmp, obj(x, y));
		}

	return redux_func(std::list<type_kernel>({ tmp }));
}

// pixel-to-pixel conversion (2 ops)
template<typename type_in1, typename type_in2, typename type_ret> map2d<type_ret> map2d_in2out(const map2d<type_in1>& in1, const map2d<type_in1>& in2, std::function<type_ret(type_in1, type_in2)> func)
{
	if (in1.width() != in2.width() || in1.height() != in2.height())
		throw_exception(map_size_exception, in1.width(), in1.height(), in2.width(), in2.height());

	map2d<type_ret> ret(in1.width(), in1.height());

	if (!func)
		throw_exception(map_function_exception);

	for (size_t y = 0; y < in1.height(); y++)
		for (size_t x = 0; x < in1.width(); x++)
			ret(x, y) = func(in1(x, y), in2(x, y));

	return ret;
};

// pixel-to-pixel conversion (2 ops - 1 const)
template<typename type_in1, typename type_in2, typename type_ret> map2d<type_ret> map2d_in2out(const map2d<type_in1>& in1, const type_in2 &in2, std::function<type_ret(type_in1,type_in2)> func)
{
	map2d<type_ret> ret(in1.width(), in1.height());

	if (!func)
		throw_exception(map_function_exception);

	for (size_t y = 0; y < in1.height(); y++)
		for (size_t x = 0; x < in1.width(); x++)
			ret(x, y) = func(in1(x, y), in2);

	return ret;
};

// pixel-to-pixel conversion (2 ops - 1 const)
template<typename type_in1, typename type_in2, typename type_ret> map2d<type_ret> map2d_in2out(const type_in2& in1, const map2d<type_in1>& in2, std::function<type_ret(type_in1, type_in2)> func)
{
	map2d<type_ret> ret(in2.width(), in2.height());

	if (!func)
		throw_exception(map_function_exception);

	for (size_t y = 0; y < in2.height(); y++)
		for (size_t x = 0; x < in2.width(); x++)
			ret(x, y) = func(in1, in2(x, y));

	return ret;
};

// region copy
template<typename type> const map2d<type> region_copy(map2d<type>& self, type value, std::shared_ptr<const map2d<bool>> mask = nullptr)
{
	if (mask != nullptr && (mask->width() != self.width() || mask->height() != self.height()))
		throw_exception(map_size_exception, self.width(), self.height(), mask->width(), mask->height());

	for (size_t y = 0; y < self.height(); y++)
		for (size_t x = 0; x < self.width(); x++)
		{
			if (mask != nullptr && !(*mask)(x, y))
				continue;

			self(x, y) = value;
		}

	return self;
};

// pixel-to-pixel conversion (1 op)
template<typename type_in> const map2d<type_in>& map2d_selfop(map2d<type_in>& self, std::function<void(type_in&)> func, std::shared_ptr<const map2d<bool>> mask = nullptr)
{
	if (mask != nullptr && (mask->width() != self.width() || mask->height() != self.height()))
		throw_exception(map_size_exception, self.width(), self.height(), mask->width(), mask->height());

	if (!func)
		throw_exception(map_function_exception);

	for (size_t y = 0; y < self.height(); y++)
		for (size_t x = 0; x < self.width(); x++)
		{
			if (mask != nullptr && !(*mask)(x, y))
				continue;

			func(self(x, y));
		}

	return self;
};

// pixel-to-pixel conversion (2 ops)
template<typename type_in1, typename type_in2> const map2d<type_in1>& map2d_selfop(map2d<type_in1>& self, const map2d<type_in2>& other, std::function<void(type_in1&, type_in2)> func, std::shared_ptr<const map2d<bool>> mask = nullptr)
{
	if (self.width() != other.width() || self.height() != other.height())
		throw_exception(map_size_exception, self.width(), self.height(), other.width(), other.height());

	if (mask != nullptr && (mask->width() != self.width() || mask->height() != self.height()))
		throw_exception(map_size_exception, self.width(), self.height(), mask->width(), mask->height());

	if (!func)
		throw_exception(map_function_exception);

	for (size_t y = 0; y < self.height(); y++)
		for (size_t x = 0; x < self.width(); x++)
		{
			if (mask != nullptr && !(*mask)(x, y))
				continue;

			func(self(x, y), other(x, y));
		}

	return self;
};

// pixel-to-pixel conversion (2 ops - 1 const)
template<typename type_in1, typename type_in2> const map2d<type_in1>& map2d_selfop(map2d<type_in1>& self, const type_in2& other, std::function<void(type_in1&, type_in2)> func, std::shared_ptr<const map2d<bool>> mask = nullptr)
{
	if (mask != nullptr && (mask->width() != self.width() || mask->height() != self.height()))
		throw_exception(map_size_exception, self.width(), self.height(), mask->width(), mask->height());

	if (!func)
		throw_exception(map_function_exception);

	for (size_t y = 0; y < self.height(); y++)
		for (size_t x = 0; x < self.width(); x++)
		{
			if (mask != nullptr && !(*mask)(x, y))
				continue;

			func(self(x, y), other);
		}

	return self;
};

#define MAP2D_IMPL_FUNC(type_in, type_ret, name, func)			static map2d<type_ret> name(const map2d<type_in>& obj)			\
																{																\
																	return map2d_in2out<type_in, type_ret>(obj, func);			\
																}

#define MAP2D_IMPL_STD_FUNC_EX(type_in, type_ret, func)			MAP2D_IMPL_FUNC(type_in, type_ret, func, [](type_in val)			\
																{																\
																	return (type_ret)std::func(val);								\
																});

#define MAP2D_IMPL_STD_FUNC(type, func)							MAP2D_IMPL_STD_FUNC_EX(type, type, func);

#define MAP2D_IMPL_OPERATOR(type_in, type_ret, op)				static map2d<type_ret> operator op(const map2d<type_in>& a, const type_in b)						\
																{																								\
																	return map2d_in2out<type_in, type_in, type_ret>(a, b, [](const type_in a, const type_in b)		\
																		{																						\
																			return (type_ret)(a op b);															\
																		});																						\
																}																								\
																																								\
																static map2d<type_ret> operator op(const type_in a, const map2d<type_in>& b)						\
																{																								\
																	return map2d_in2out<type_in, type_in, type_ret>(a, b, [](const type_in a, const type_in b)		\
																		{																						\
																			return (type_ret)(a op b);															\
																		});																						\
																}																								\
																																								\
																static map2d<type_ret> operator op(const map2d<type_in>& a, const map2d<type_in>& b)				\
																{																								\
																	return map2d_in2out<type_in, type_in, type_ret>(a, b, [](const type_in a, const type_in b)		\
																		{																						\
																			return (type_ret)(a op b);															\
																		});																						\
																}

#define MAP2D_IMPL_SELF_OPERATOR(type_in, op)					static const map2d<type_in>& operator op(map2d<type_in>& a, const type_in b)						\
																{																								\
																	return map2d_selfop<type_in, type_in>(a, b, [](type_in& a, const type_in b)						\
																		{																						\
																			a op b;																				\
																		});																						\
																}																								\
																																								\
																static const map2d<type_in>& operator op(map2d<type_in>& a, const map2d<type_in>& b)				\
																{																								\
																	return map2d_selfop<type_in, type_in>(a, b, [](type_in& a, const type_in b)						\
																		{																						\
																			a op b;																				\
																		});																						\
																}
