#pragma once

#include "../utils/utils.h"
#include "../debug/safe.h"
#include "../scope/base.h"
#include "../scope/ticket.h"

#include "callback.h"

#include <memory>
#include <vector>
#include <unordered_map>

namespace event::notify
{
	/* base class */
	template<typename ... params> class base
	{
	public:
		using observant_t = event::callback<void, params ...>;
		using observant_list_t = std::vector<observant_t>;

		// register observers
		bool listen(int event, observant_t obj)
		{
			// skip if callback is invalid
			if (!obj.is_valid())
			{
				_warning("invalid observer, listenning aborted!");

				return false;
			}

			// check if entries already exists
			auto it = this->m_observers.find(event);

			// add to list if event is already listenned
			if (it != this->m_observers.end())
			{
				it->second.push_back(obj);

				return true;
			}

			// create if no data yet
			this->m_observers.emplace(std::make_pair(event, observant_list_t({ obj })));

			return true;
		}

		// register from function and scope
		bool listen(int event, std::function<void(params ...)> func, const scope::ticket::handle& obj)
		{
			return listen(event, event::callback(func, obj));
		}

		// register from function and scope
		bool listen(int event, std::function<void(params ...)> func, const scope::ticket& obj)
		{
			return listen(event, func, obj.create_handle());
		}

		// register from object observer
		// bool listen(int event, std::function<void(event::scope::base*, params ...)> func, event::scope::base *obj)
		bool listen(int event, std::function<void(params ...)> func, scope::base *obj)
		{
			if (obj == nullptr)
			{
				_warning("null object, listenning aborted!");

				return false;
			}

			return listen(event, func, obj->get_scope_ticket());
		}

	protected:

		// called after every notifications
		virtual void on_notify(int) {}

		// notify event
		auto notify(int event, params ... args)
		{
			size_t cnt = 0;

			// open list
			auto it = this->m_observers.find(event);

			// browse list and call functions
			if (it != this->m_observers.end())
			{
				for (auto& v : it->second)
					if (v.is_valid())
					{
						// try/catch because object may have disappeared in the meanwhile
						try
						{
							v.call(args ...);

							MAKE_INC(cnt, (size_t)1);
						}
						catch (...) {}
					}
			}

			// notify owner object
			on_notify(event);

			// return number of observer that were effectively notified
			return cnt;
		}

	private:

		// callbacks are stored in map for fast access
		std::unordered_map<int, observant_list_t> m_observers;
	};
};

#define _SELF(class,func)	std::bind(&class::func, this)
