#pragma once

#include "../utils/utils.h"
#include "../debug/evemon.h"
#include "../exception.h"

#include "exception.h"

#include <string>
#include <climits>

// safe_cast_exception exception class
class safe_cast_exception : public exception::base
{
public:
    virtual std::string to_string(void) const override
    {
        return "safe_cast operation failed!";
    }
};

// generic safe cast will not compile
template<typename type1, typename type2> type1 safe_cast(const type2 value)
{
    auto ret = (type1)value;

    if ((type2)ret != value)
        throw_exception(safe_cast_exception);

    return ret;
}

#define MAKE_SIZE_T(value) safe_cast<size_t>(value)
#define MAKE_PTRDIFF_T(value) safe_cast<ptrdiff_t>(value)
#define MAKE_UINTPTR_T(value) safe_cast<uintptr_t>(value)

#define MAKE_WCHAR_T(value) safe_cast<wchar_t>(value)

#define MAKE_CHAR(value) safe_cast<char>(value)
#define MAKE_SHORT(value) safe_cast<short>(value)
#define MAKE_INT(value) safe_cast<int>(value)
#define MAKE_LONG(value) safe_cast<long>(value)
#define MAKE_LONGLONG(value) safe_cast<long long>(value)

#define MAKE_UCHAR(value) safe_cast<unsigned char>(value)
#define MAKE_USHORT(value) safe_cast<unsigned short>(value)
#define MAKE_UINT(value) safe_cast<unsigned int>(value)
#define MAKE_ULONG(value) safe_cast<unsigned long>(value)
#define MAKE_ULONGLONG(value) safe_cast<unsigned long long>(value)

#define MAKE_INT8(value) safe_cast<int8_t>(value)
#define MAKE_INT16(value) safe_cast<int16_t>(value)
#define MAKE_INT32(value) safe_cast<int32_t>(value)
#define MAKE_INT64(value) safe_cast<int64_t>(value)
#define MAKE_INT128(value) safe_cast<int128_t>(value)

#define MAKE_UINT8(value) safe_cast<uint8_t>(value)
#define MAKE_UINT16(value) safe_cast<uint16_t>(value)
#define MAKE_UINT32(value) safe_cast<uint32_t>(value)
#define MAKE_UINT64(value) safe_cast<uint64_t>(value)
#define MAKE_UINT128(value) safe_cast<uint128_t>(value)

// safe_add_exception exception class
class safe_add_exception : public exception::base
{
public:
    virtual std::string to_string(void) const override
    {
        return "safe addition operation failed!";
    }
};

template<typename type> type unsigned_max_val(void)
{
	throw_exception(template_func_not_impl_exception, typeid(type).name());
};

#define DEFINE_UINT_MAXVAL(type, MaxValue)  template<> inline type unsigned_max_val(void)     \
                                            {                                                 \
                                                return MaxValue;                              \
                                            }

DEFINE_UINT_MAXVAL(unsigned char, UCHAR_MAX)
DEFINE_UINT_MAXVAL(unsigned short, USHRT_MAX)
DEFINE_UINT_MAXVAL(unsigned int, UINT_MAX)
DEFINE_UINT_MAXVAL(unsigned long, ULONG_MAX)
DEFINE_UINT_MAXVAL(unsigned long long, ULLONG_MAX)

template<typename type> type safe_add(const type val_a, const type val_b)
{
    if (val_a >= unsigned_max_val<type>() - val_b)
        throw_exception(safe_add_exception);
    
    return val_a + val_b;
}

#define MAKE_ADD(val_a, val_b) safe_add(val_a, val_b)
#define MAKE_INC(val_a, val_b) val_a = MAKE_ADD(val_a, val_b)

// safe_subtract_exception exception class
class safe_subtract_exception : public exception::base
{
public:
    virtual std::string to_string(void) const override
    {
        return "safe subtraction operation failed!";
    }
};

template<typename type> type safe_sub(const type val_a, const type val_b)
{
    if (val_a < val_b)
        throw_exception(safe_subtract_exception);

    return val_a - val_b;
}

#define MAKE_SUB(val_a, val_b) safe_sub(val_a, val_b)
#define MAKE_DEC(val_a, val_b) val_a = MAKE_SUB(val_a, val_b)

// safe_multiply_exception exception class
class safe_multiply_exception : public exception::base
{
public:
    virtual std::string to_string(void) const override
    {
        return "safe multiply operation failed!";
    }
};

// safe multiplication
template<typename type> type safe_mult(const type val_a, const type val_b)
{
    auto countbits = [](type value)
    {
        size_t i = 0;

        while (((type)1 << i) < value && (i+1) < 8 * sizeof(type))
            i++;

        return i;
    };

    if (countbits(val_a) + countbits(val_b) > 8 * sizeof(type))
        throw_exception(safe_multiply_exception);

    return val_a * val_b;
}

#define MAKE_MULT(val_a, val_b) safe_mult(val_a, val_b)

// safe_address_exception exception class
class safe_address_exception : public exception::base
{
public:
    virtual std::string to_string(void) const override
    {
        return "safe address operation failed!";
    }
};

// safe address computation
template<typename type> type* safe_address(const type *base, const uintptr_t offset)
{
    if ((uintptr_t)base >= UINTPTR_MAX - offset)
        throw_exception(safe_address_exception);

    return (type*)((unsigned char*)base + offset);
}

#define MAKE_ADDRESS(base, ofs) safe_address(base, MAKE_UINTPTR_T(ofs))

// increment pointer if not null
template<typename type> type* safe_incptr(const type* base, const uintptr_t offset)
{
    if (base == nullptr)
        return nullptr;

    return MAKE_ADDRESS(base, MAKE_MULT(offset, sizeof(type)));
}

#define MAKE_AT(base, ofs) safe_incptr(base, ofs)

// macro that are usefull
#define MAKE_XY(x, y, width)	(MAKE_ADD(x, MAKE_MULT(y, width)))
