#pragma once

#include <stdio.h>
#include <filesystem>

#include "../utils/utils.h"
#include "../debug/evemon.h"
#include "../debug/exdebug.h"
#include "../debug/exception.h"
#include "../resources/uid.h"
#include "../storage/file.h"
#include "../storage/storage.h"
#include "../hash/sha2.h"

#include "node.h"
#include "command.h"
#include "exception.h"

#include <list>
#include <memory>
#include <string>
#include <filesystem>
#include <stack>

namespace version
{
	// cleanup filename
	static std::string cleanup(std::string filename)
	{
		// replace any non-alphanumeric character by '-'
		std::for_each(filename.begin(), filename.end(), [](char c)
			{
				if (isalnum(c))
					return c;

				return '-';
			});

		return filename;
	}

	// convert id to path
	static std::string id2path(const std::string& project_name, const std::string& commit_id)
	{
		return build_path({ cleanup(project_name), "COMMITS", cleanup(commit_id) });
	}

	// extract id from path
	static std::string path2id(const std::string& filename)
	{
		auto id = split_file_parts(filename).file;

		// valid commits are hexadecimal representation of 256 bits strings
		if (id.length() != 64)
			throw_exception(erroneous_commit_id_exception, filename);

		return id;
	}

	// save node
	static std::string save(const std::string& project_name, std::shared_ptr<version::node> p)
	{
		// pack to buffer
		storage::container file;
		file.push_back(p);

		auto buffer = file.pack('SPC0');

		// compute id
		auto id = storage::buf2hex(sha256(buffer));

		// save to file
		auto filename = id2path(project_name, id);

		storage::save_buffer_to_file(filename, buffer);

		// change permission to 0440
		std::filesystem::permissions(filename, std::filesystem::perms::owner_read | std::filesystem::perms::group_read, std::filesystem::perm_options::replace);

		// return filename
		return id;
	}

	// set commit
	static void set_as(const std::string& filename, const std::string& commit_id)
	{
		// open file
		FILE* p = nullptr;

		fopen_s(&p, filename.c_str(), "wt+");

		if (p == nullptr)
			throw_exception(cannot_open_file_exception, filename);

		// write commit id
		try
		{
			fputs(commit_id.c_str(), p);
		}
		catch (...)
		{
			// close file and rethrow
			fclose(p);

			throw;
		}

		// close file
		fclose(p);
	}

	// set commit as HEAD
	static void set_head(const std::string& project_name, const std::string& user_name, const std::string& commit_id)
	{
		// build filename
		auto filename = build_path({ cleanup(project_name), "HEADS", cleanup(user_name) });

		// set
		set_as(filename, commit_id);

		// change permission to 0640
		std::filesystem::permissions(filename, std::filesystem::perms::owner_read | std::filesystem::perms::owner_write | std::filesystem::perms::group_read, std::filesystem::perm_options::replace);
	}

	// set commit as TAG
	static void set_tag(const std::string& project_name, const std::string& tag_name, const std::string& commit_id)
	{
		// build filename
		auto filename = build_path({ cleanup(project_name), "TAGS", cleanup(tag_name) });

		// set
		set_as(filename, commit_id);

		// change permission to 0440
		std::filesystem::permissions(filename, std::filesystem::perms::owner_read | std::filesystem::perms::group_read, std::filesystem::perm_options::replace);
	}

	// resolve
	static std::string resolve(const std::string& filename)
	{
		// check filename
		if (!std::filesystem::exists(filename))
			throw_exception(unknown_commit_exception, filename);

		// load file
		FILE* p = nullptr;

		fopen_s(&p, filename.c_str(), "rt");

		if (p == nullptr)
			throw_exception(cannot_open_file_exception, filename);

		// read file and get content
		char buffer[128];

		fgets(buffer, sizeof(buffer), p);

		fclose(p);

		// return commit
		return std::string(buffer);
	}

	// resolve HEAD
	static std::string resolve_head(const std::string& project_name, const std::string& user_name)
	{
		return resolve(build_path({ cleanup(project_name), "HEADS", cleanup(user_name) }));
	}

	// resolve TAG
	static std::string resolve_tag(const std::string& project_name, const std::string& tag_name)
	{
		// build filename
		return resolve(build_path({ cleanup(project_name), "TAGS", cleanup(tag_name) }));
	}

	// list all types
	static std::list<std::string> list_all(const std::string& project_name, const std::string& type)
	{
		auto dirname = build_path({ cleanup(project_name), cleanup(type) });

		std::list<std::string> list;

		for (auto& curr : std::filesystem::directory_iterator(dirname))
		{
			if (!curr.is_regular_file())
				continue;

			list.push_back(split_file_parts(curr.path().string()).file);
		}

		return list;
	}

	// list all tags
	static std::list<std::string> list_tags(const std::string& project_name)
	{
		return list_all(project_name, "TAGS");
	}

	// list all heads
	static std::list<std::string> list_heads(const std::string& project_name)
	{
		return list_all(project_name, "HEADS");
	}

	// list all commits
	static std::list<std::string> list_commits(const std::string& project_name)
	{
		return list_all(project_name, "COMMITS");
	}

	// create node
	static auto create_node(const std::string& user_name, std::list<std::string> parents)
	{
		return std::make_shared<version::node>(user_name, parents);
	}

	// create directory structure and root node
	static void init(const std::string& project_name, const std::string& user_name)
	{
		// skip if directory already exits
		if (std::filesystem::exists(project_name))
			throw_exception(project_already_exists_exception);

		// create directory structure
		std::filesystem::create_directory(project_name);

		std::filesystem::create_directory(build_path({ project_name, "COMMITS" }));
		std::filesystem::create_directory(build_path({ project_name, "HEADS" }));
		std::filesystem::create_directory(build_path({ project_name, "TAGS" }));

		// add write permission to other group members
		std::filesystem::permissions(build_path({ project_name, "COMMITS" }), std::filesystem::perms::group_write, std::filesystem::perm_options::add);
		std::filesystem::permissions(build_path({ project_name, "HEADS" }), std::filesystem::perms::group_write, std::filesystem::perm_options::add);
		std::filesystem::permissions(build_path({ project_name, "TAGS" }), std::filesystem::perms::group_write, std::filesystem::perm_options::add);
	}

	// resolve a tag or commit to commit id
	static std::string _resolve_commit_id(const std::string& project_name, const std::string& user_name, const std::string& commit_or_tag)
	{
		// check for HEAD
		if(str_begins_with(commit_or_tag, "HEAD/"))
			return resolve_head(project_name, commit_or_tag.c_str() + 5);

		// special case for HEAD
		if (strcmp(commit_or_tag.c_str(), "HEAD") == 0)
			return resolve_head(project_name, user_name);

		// check if it is a valid commit id
		{
			auto filename = id2path(project_name, commit_or_tag);

			if (std::filesystem::exists(filename))
				return commit_or_tag;
		}

		// check for TAG
		{
			auto filename = build_path({ cleanup(project_name), "TAGS", cleanup(commit_or_tag) });

			if (std::filesystem::exists(filename))
				return resolve_tag(project_name, commit_or_tag);
		}

		// check for partial match
		{
			std::string short_id;

			// get substring
			if (str_ends_with(commit_or_tag, "..."))
				short_id = commit_or_tag.substr(0, commit_or_tag.length() - 3);
			else
				short_id = commit_or_tag;

			// list all corresponding files
			auto dirname = build_path({ cleanup(project_name), "COMMITS" });

			std::list<std::string> list;

			for (auto& curr : std::filesystem::directory_iterator(dirname))
			{
				if (!curr.is_regular_file())
					continue;

				auto filename = path2id(curr.path().string());

				if (str_begins_with(filename, short_id))
					list.push_back(filename);
			}

			// throw error if more than one
			if (list.size() > 1)
				throw_exception(ambiguous_name_exception, commit_or_tag);

			// return filename if one
			if (list.size() == 1)
				return list.front();
		}

		// nothing was found
		throw_exception(unknown_commit_exception, commit_or_tag);
	}

	// resolve decorated transactions
	static std::string resolve_commit_id(const std::string& project_name, const std::string& user_name, const std::string& commit_or_tag)
	{
		// check for last symbol ~ position
		auto pos = commit_or_tag.find_last_of('~');

		// normal behavior if no decorators are found
		if(pos == std::string::npos)
			return _resolve_commit_id(project_name, user_name, commit_or_tag);

		// split between transaction name and parent id
		auto new_commit_id = commit_or_tag.substr(0, pos);
		auto index = commit_or_tag.substr(pos + 1);

		// resolve this new commit id
		auto id = resolve_commit_id(project_name, user_name, new_commit_id);

		// load file
		version::proxy<version::node> program(id2path(project_name, id));

		int parent_id = 0;

		if (index.length() > 0)
		{
			parent_id = atol(index.c_str()) - 1;

			if (parent_id < 0)
				throw_exception(no_parent_exception, id, parent_id);
		}

		// get parent
		auto parents = program->get_parents();

		for(auto i=0;i<parent_id;i++)
		{
			if (parents.empty())
				throw_exception(no_parent_exception, id, parent_id + 1);

			parents.pop_front();
		}

		if (parents.empty())
			throw_exception(no_parent_exception, id, parent_id + 1);

		// get parent id
		return path2id(parents.front());
	}

	// set HEAD to given commit
	static std::string checkout(const std::string& project_name, const std::string& user_name, const std::string& commit_or_tag)
	{
		// resolve commit id
		auto commit_id = resolve_commit_id(project_name, user_name, commit_or_tag);

		// set head
		set_head(project_name, user_name, commit_id);

		// return full filename
		return commit_id;
	}

	// get equivalent short name
	std::string get_short_name(const std::string& project_name, const std::string& commit_id, size_t len=5)
	{
		// check if it is a valid commit id
		{
			auto filename = id2path(project_name, commit_id);

			if (!std::filesystem::exists(filename))
				throw_exception(unknown_commit_exception, commit_id);
		}

		// get shortned version
		auto short_id = commit_id.substr(0, len);
		
		// check if there is more than one match
		{
			// list all corresponding files
			auto dirname = build_path({ cleanup(project_name), "COMMITS" });

			std::list<std::string> list;

			for (auto& curr : std::filesystem::directory_iterator(dirname))
			{
				if (!curr.is_regular_file())
					continue;

				auto filename = path2id(curr.path().string());

				if (str_begins_with(filename, short_id))
					list.push_back(filename);
			}

			// return name if only one
			if (list.size() == 1)
			{
				if ((short_id.length() + 3) >= commit_id.length())
					return commit_id;

				return short_id + "...";
			}
		}

		// not matching, try with one more character
		return get_short_name(project_name, commit_id, len + 1);
	}
};
