#pragma once

#include "../storage/base.h"
#include "../instancers/instancer.h"

#include "command.h"
#include "exception.h"

#include <memory>
#include <string>
#include <list>

namespace version
{
	/* transaction class */
	class transaction : public storage::base
	{
	public:

		// build typename
		virtual std::string get_typename(void) const override
		{
			return "version::transaction";
		}

		// push to storage object
		virtual void push(storage::fs::object_base& container) const
		{
			// write all commands
			for (auto& p : this->m_commands)
			{
				if (p == nullptr)
					continue;

				// write as sub-object
				auto sub_object = container.create_sub_object("", "m_commands");

				if (sub_object == nullptr)
					continue;

				p->push(*sub_object);
			}

			// set typename
			container.set_type_name(get_typename());
		}

		// pop from storage object
		virtual void pop(const storage::fs::object_base& container)
		{
			// clear current commands list
			this->m_commands.clear();

			// check typename
			if (!is_type_name(container.get_type_name(), get_typename()))
				throw_exception(wrong_type_exception, get_typename(), container.get_type_name());

			// browse all children
			auto children = container.get_children();

			for (auto& p : children)
			{
				if (p == nullptr)
					continue;

				// we expect the variable name to be m_commands
				if (p->get_var_name() != "m_commands")
					continue;

				// we expect the type to be version::command
				auto obj = create_instance<version::command>(p->get_type_name());

				if (obj == nullptr)
					continue;

				obj->pop(*p);

				this->m_commands.push_back(obj);
			}
		}

		// apply all commands
		void run(void) const
		{
			for (auto& p : this->m_commands)
				if (p != nullptr)
					p->run();
		}

		// retrieve log information
		std::list<std::string> log(void) const
		{
			std::list<std::string> ret;

			for (auto& p : this->m_commands)
				if (p != nullptr)
					ret.push_back(p->log());

			return ret;
		}

		// push a command
		void push(std::shared_ptr<command> p)
		{
			this->m_commands.push_back(p);
		}

		// pop a command
		void pop(void)
		{
			if (this->m_commands.empty())
				throw_exception(empty_transaction_exception);

			this->m_commands.pop_back();
		}

		// clear list
		void clear(void)
		{
			this->m_commands.clear();
		}

		// check if empty
		bool is_empty(void) const
		{
			return this->m_commands.empty();
		}

		// merge transaction
		void merge(const transaction& other)
		{
			for (auto& p : other.m_commands)
			{
				if (p == nullptr)
					continue;

				// skip if command already exists
				if (has(p->uid()))
					continue;

				push(p);
			}
		}

		// check if transaction has given command uid
		bool has(const std::string& uid) const
		{
			for (auto& p : this->m_commands)
			{
				if (p == nullptr)
					continue;

				if (p->uid() == uid)
					return true;
			}

			return false;
		}

	private:
		std::list<std::shared_ptr<command>> m_commands;
	};
};

using version_transaction_t = version::transaction;