#pragma once

#include "../hash/sha2.h"
#include "../storage/file.h"
#include "../storage/storage.h"
#include "../storage/ext/string.h"

#include "node.h"
#include "exception.h"

namespace version
{
	/* proxy template class */
	template<typename type> class proxy : public storage::base
	{
	public:
		IMPLEMENT_STORAGE;

		// reset pointer on file reload to force reconstruction of pointer
		virtual void custom_pop(const storage::fs::object_base&) override
		{
			this->m_pointer.reset();
		}

		// construct from name
		proxy(const std::string& name = "")
		{
			this->m_name = name;
		}

		// copy ctor
		proxy(const proxy& other)
		{
			this->operator=(other);
		}

		// move ctor
		proxy(proxy&& other) noexcept
		{
			this->operator=(std::move(other));
		}

		// copy operator
		const proxy<type>& operator=(const proxy<type>& other)
		{
			// reset pointer to force reload
			this->m_pointer.reset();

			// copy name
			this->m_name = other.m_name;

			return *this;
		}

		// move operator
		const proxy<type>& operator=(proxy<type>&& other) noexcept
		{
			this->m_pointer = std::move(other.m_pointer);
			this->m_name = std::move(other.m_name);

			return *this;
		}

		// return name
		const std::string& name(void) const
		{
			return this->m_name;
		}

		// return pointer
		std::shared_ptr<type> operator->(void)
		{
			return ptr();
		}

		// return pointer (const version)
		std::shared_ptr<const type> operator->(void) const
		{
			return const_cast<proxy<type>*>(this)->ptr();
		}

		// return reference
		type& operator*(void)
		{
			return *ptr();
		}

		// return reference (const version)
		const type& operator*(void) const
		{
			return *(const_cast<proxy<type>*>(this)->ptr());
		}

	private:

		// get pointer
		std::shared_ptr<type>& ptr(void)
		{
			// load file if pointer is null
			if (this->m_pointer == nullptr)
			{
				_debug("loading proxy %s...", this->m_name.c_str());

				// load buffer
				auto buffer = storage::load_buffer_from_file(this->m_name);

				// check sha256
				auto cksum = storage::buf2hex(sha256(buffer));

				if (cksum != split_file_parts(this->m_name).file)
					throw_exception(corrupted_file_exception, this->m_name);

				// unpack
				auto _create = [](const std::string& type_name)
				{
					auto parts = split_file_parts(type_name);

					if (!is_type_name<type>(parts.file))
						throw_exception(unknown_typename_exception, type_name);

					return std::static_pointer_cast<storage::base>(std::make_shared<type>());
				};

				storage::container file;

				file.unpack(buffer, _create);

				// load content
				if (file.size() != 1)
					throw_exception(invalid_file_exception, this->m_name);

				std::string owner, name;
				std::shared_ptr<storage::base> obj;

				std::tie(owner, name, obj) = file.front();

				this->m_pointer = std::static_pointer_cast<type>(obj);
			}

			// throw exception if pointer is still null
			if (this->m_pointer == nullptr)
				throw_exception(null_pointer_exception);

			// return pointer
			return this->m_pointer;
		}

	private:
		storage::ext::string m_name;

		std::shared_ptr<type> m_pointer;
	};

	/* proxy_list template class */
	template<typename type> class proxy_list : public std::list<proxy<type>>, public storage::base
	{
	public:
		using std::list<proxy<type>>::list;
		using std::list<proxy<type>>::operator=;

		// construct typename
		virtual std::string get_typename(void) const override
		{
			return "version::proxy_list<" + get_name_from_type<type>() + ">";
		}

		// push to storage object
		virtual void push(storage::fs::object_base& container) const
		{
			// add all proxies as subobjects
			for (auto it = this->begin(); it != this->end(); it++)
			{
				auto sub_object = container.create_sub_object("", "");

				if (sub_object == nullptr)
					continue;

				it->push(*sub_object);
			}

			// set typename
			container.set_type_name(get_typename());
		}

		// pop from storage object
		virtual void pop(const storage::fs::object_base& container)
		{
			// clear current list
			this->clear();

			// check typename
			if (!is_type_name(container.get_type_name(), get_typename()))
				throw_exception(wrong_type_exception, get_typename(), container.get_type_name());

			// browse all children
			auto children = container.get_children();

			for (auto& p : children)
			{
				if (p == nullptr)
					continue;

				// we expect the type to be proxy<type>
				proxy<type> obj;

				obj.pop(*p);

				this->push_back(obj);
			}
		}
	};
};
