#pragma once

#include "../storage/storage.h"
#include "../storage/ext/string.h"

#include "command.h"
#include "transaction.h"
#include "proxy.h"

#include <memory>
#include <string>
#include <list>

namespace version
{
	/* node class */
	class node : public storage::base
	{
	public:
		IMPLEMENT_STORAGE;

		// default ctor for storage retrieval
		node(void)
		{
			this->m_owner = "?";
		}

		// build node from owner name and list of parents
		node(const std::string& owner, std::list<std::string> parents)
		{
			this->m_owner = owner;

			replace_parents(parents);
		}

		// replace parents
		void replace_parents(std::list<std::string> parents)
		{
			// clear previous
			this->m_parents.clear();

			// add all parents
			for (auto& v : parents)
				this->m_parents.push_back(proxy<node>(v));
		}

		// retrieve list of parents name
		std::list<std::string> get_parents(void) const
		{
			std::list<std::string> ret;

			for (auto& p : this->m_parents)
				ret.push_back(p.name());

			return ret;
		}

		// retrieve all commands (local)
		const transaction& get_commands(void) const
		{
			return this->m_commands;
		}

		// clone node
		std::shared_ptr<node> clone(void) const
		{
			auto p = std::make_shared<node>();

			p->m_owner = this->m_owner;
			p->m_parents = this->m_parents;
			p->m_commands = this->m_commands;

			return p;
		}

		// return log information about node and its parents
		std::list<std::string> log(void) const
		{
			std::list<std::string> ret;

			// log all parents first
			for(auto& p : this->m_parents)
				ret.splice(ret.end(), p->log());

			// log local commands next
			auto cmds = this->m_commands.log();

			if (cmds.size() > 0)
			{
				ret.push_back("Changes authored by " + this->m_owner + ":");

				ret.splice(ret.end(), cmds);
			}

			return ret;
		}

		// push a command
		bool push(std::shared_ptr<command> p)
		{
			// skip if null
			if (p == nullptr)
				return false;

			// skip if command uid already exists
			if (has(p->uid()))
				return false;

			// push command and return OK
			this->m_commands.push(p);

			return true;
		}

		// check if command already exists
		bool has(const std::string& name) const
		{
			// return true if local commands include the name
			if (this->m_commands.has(name))
				return true;

			// return true if one of the parents has the command (recursive)
			for (auto& p : this->m_parents)
				if (p->has(name))
					return true;

			// otherelse return false
			return false;
		}

		// check if node has some commands to be applied
		bool has_transactions(void) const
		{
			return !this->m_commands.is_empty();
		}

		// pop command (undo)
		void pop(void)
		{
			this->m_commands.pop();
		}

		// integrate all transactions
		transaction integrate(void) const
		{
			// push current commands last
			transaction ret = this->m_commands;

			// append all parents transaction on front
			for (auto it = this->m_parents.rbegin(); it != this->m_parents.rend(); it++)
			{
				auto p = *it;

				auto tmp = p->integrate();

				tmp.merge(ret);

				ret = tmp;
			}

			// return concatenated transactions
			return ret;
		}

	private:
		proxy_list<node> m_parents;
		transaction m_commands;
		storage::ext::string m_owner;
	};
};

using version_node_t = version::node;