#pragma once

#include "../debug/exdebug.h"
#include "../utils/singleton.h"

#include "type.h"

#include <unordered_map>
#include <cstdlib>

namespace endian
{
	// return true if little endian
	constexpr bool is_little_endian(void)
	{
		return ((unsigned long)'ABCD') == 0x41424344;
	}

	// pack variable in memory using little endian ordering
	template<typename type> void pack(unsigned char* mem, const type& value, const size_t size = sizeof(type))
	{
		if (mem == nullptr)
			return;

		if (size != sizeof(type))
			throw_exception(wrong_size_exception, size, sizeof(type));

		if constexpr (is_little_endian())
			std::memcpy(mem, &value, sizeof(type));
		else
		{
			const unsigned char* p = (const unsigned char*)&value;

			for (size_t i = 0; i < size; i++)
				mem[size - i - 1] = p[i];
		}
	}

	// unpack variable using little endian ordering
	template<typename type> void unpack(const unsigned char* mem, type& value, const size_t size = sizeof(type))
	{
		if (mem == nullptr)
			return;

		if (size != sizeof(type))
			throw_exception(wrong_size_exception, size, sizeof(type));

		if constexpr (is_little_endian())
			std::memcpy(&value, mem, size);
		else
		{
			const unsigned char* p = (const unsigned char*)&value;

			for (size_t i = 0; i < size; i++)
				p[i] = mem[size - i - 1];
		}
	}

	/* endianness manager */
	class manager : public singleton<endian::manager>
	{
		friend class singleton<endian::manager>;

	private:

		/* handler base class */
		class handler_base
		{
		public:
			virtual void read(const void* src, void* dst, const size_t size) const = 0;
			virtual void write(const void* src, void* dst, const size_t size) const = 0;
		};

		/* specific type handle */
		template<typename type> class handler : public handler_base
		{
		public:
			virtual void read(const void* src, void* dst, const size_t size) const override
			{
				unpack((const unsigned char*)src, *(type*)dst, size);
			}

			virtual void write(const void* src, void* dst, const size_t size) const override
			{
				pack((unsigned char*)dst, *(type*)src, size);
			}
		};

	public:

		// add type to list
		template<typename type> void register_type(const std::string& cname)
		{
			if (has_type(cname))
				return;

			this->m_list.emplace(std::make_pair(cname, std::make_unique<handler<type>>()));
		}

		// return true if type exists
		bool has_type(const std::string& cname) const
		{
			return this->m_list.find(cname) != this->m_list.end();
		}

		// read type
		void read(const std::string& tname, const void* src, void* dst, const size_t size) const
		{
			for (auto& v : this->m_list)
			{
				if (__ISNULLPTR(v.second))
					continue;

				if (!is_type_name(v.first, tname))
					continue;

				v.second->read(src, dst, size);

				return;
			}

			throw_exception(unknown_typename_exception, tname);
		}

		// write type
		void write(const std::string& tname, const void* src, void* dst, const size_t size) const
		{
			for (auto& v : this->m_list)
			{
				if (__ISNULLPTR(v.second))
					continue;

				if (!is_type_name(v.first, tname))
					continue;

				v.second->write(src, dst, size);

				return;
			}

			throw_exception(unknown_typename_exception, tname);
		}

	private:
		std::unordered_map<std::string, std::unique_ptr<handler_base>> m_list;
	};

	/* type_auto_register class */
	template<typename type> class auto_register
	{
	public:
		auto_register(const std::string& ttype)
		{
			get_instance<endian::manager>()->register_type<type>(ttype);
		}
	};

	// read from name
	static void read(const std::string& tname, const void* src, void* dst, const size_t size)
	{
		get_instance<endian::manager>()->read(tname, src, dst, size);
	}

	// write from name
	static void write(const std::string& tname, const void* src, void* dst, const size_t size)
	{
		get_instance<endian::manager>()->write(tname, src, dst, size);
	}
};

// convert host to file
template<typename type> type htof(const type value)
{
	type ret;

	endian::pack((unsigned char*)&ret, value);

	return ret;
}

// convert file to host
template<typename type> type ftoh(const type value)
{
	type ret;

	endian::unpack((const unsigned char*)&value, ret);

	return ret;
}

// register typename
#define REGISTER_ENDIANTYPE_EX(class, string, var)	endian::auto_register<class> register_endiantype_##var(string);

#define REGISTER_ENDIANTYPE(type)					REGISTER_ENDIANTYPE_EX(type, #type, type);