#pragma once

#include "../../utils/utils.h"
#include "../../debug/exception.h"
#include "../../debug/evemon.h"
#include "../../debug/exdebug.h"
#include "../../debug/safe.h"

#include "../buffer.h"

#include "exception.h"

#include <vector>
#include <string>
#include <list>
#include <memory>

namespace storage::fs
{
	/* variable base */
	struct var_base_s
	{
		virtual ~var_base_s(void) {}

		std::string owner;
		std::string name;
		std::string type;
	};

	// object class
	class object_base
	{
	public:
		object_base(void) {}

		object_base(const std::string& owner, const std::string& var)
		{
			this->m_owner_name = owner;
			this->m_var_name = var;
		}

		object_base(const object_base&) = delete;
		object_base(object_base&&) = delete;

		virtual ~object_base(void) {}

		void clear(void)
		{
			this->m_children.clear();
			this->m_vars.clear();
		}

		// return handle to children list for raw loaders
		std::list< std::shared_ptr<const object_base> > get_children(void) const
		{
			std::list< std::shared_ptr<const object_base> > ret;

			for (auto p : this->m_children)
				ret.push_back(p);

			return ret;
		}

		// return all vars
		std::list< std::shared_ptr<const var_base_s> > get_vars(void) const
		{
			std::list< std::shared_ptr<const var_base_s> > ret;

			for (auto p : this->m_vars)
				ret.push_back(p);

			return ret;
		}

		// set var name
		void set_var_name(const std::string& name)
		{
			this->m_var_name = name;
		}

		// get var name
		std::string get_var_name(void) const
		{
			return this->m_var_name;
		}

		// set owner name
		void set_owner_name(const std::string& name)
		{
			this->m_owner_name = name;
		}

		// get owner name
		std::string get_owner_name(void) const
		{
			return this->m_owner_name;
		}

		// set type name
		void set_type_name(const std::string& name)
		{
			this->m_type_name = name;
		}

		// get type name
		std::string get_type_name(void) const
		{
			return this->m_type_name;
		}

		// get sub object from list
		std::shared_ptr<object_base> get_sub_object(const std::string& owner, const std::string& name) const
		{
			// browse list
			for (auto p : this->m_children)
			{
				if (p == nullptr)
					continue;

				// skip if different
				if (p->m_var_name != name || p->m_owner_name != owner)
					continue;

				return p;
			}

			return nullptr;
		}

		// create child
		std::shared_ptr<object_base> create_sub_object(const std::string& owner, const std::string& var)
		{
			auto sub_object = _create(owner, var);

			if (__ISNULLPTR(sub_object))
				throw_exception(memory_allocation_exception, sizeof(this));

			this->m_children.emplace_back(sub_object);

			return sub_object;
		}

		// add variable to list
		virtual void push_var(const std::string& owner, const std::string& name, const std::string& type, size_t bytes_size, void* mem) = 0;

		// read variable from list
		virtual bool pop_var(const std::string& owner, const std::string& name, const std::string& type, size_t bytes_size, void* mem) const = 0;

		// pack data into a buffer
		virtual storage::buffer pack(void) const = 0;

		// unpack data
		virtual void unpack(const storage::buffer& obj) = 0;

	protected:
		virtual std::shared_ptr<object_base> _create(const std::string& owner, const std::string& var) const = 0;

		// return all vars
		const std::list< std::shared_ptr<var_base_s> >& _get_vars(void) const
		{
			return this->m_vars;
		}

		void add_variable(std::shared_ptr<var_base_s> var)
		{
			this->m_vars.emplace_back(std::move(var));
		}

		void assert_string(const char* base, size_t offset, size_t max_size) const
		{
			if (offset >= max_size)
				throw_exception(buffer_read_exception);

			const char* p = MAKE_ADDRESS(base, offset);

			while (*(p++) != '\0')
				if (offset++ > max_size)
					throw_exception(buffer_read_exception);
		}

	private:
		std::string m_owner_name;
		std::string m_var_name;
		std::string m_type_name;

		std::list< std::shared_ptr<var_base_s> > m_vars;
		std::list< std::shared_ptr<object_base> > m_children;
	};
};