#pragma once

#include "../../debug/exception.h"
#include "../../encode/rle0.h"
#include "../../encode/rle8.h"

#include "../buffer.h"

#include "exception.h"

#include <tuple>

#define FLAG_ENABLE_RLE8			(1 << 0)
#define FLAG_ENABLE_RLE0			(1 << 1)

namespace storage::fs
{
	// type of encoding
	enum
	{
		ENCODING_NONE = 0,
		ENCODING_RLE8 = 1,
		ENCODING_RLE0 = 2,
	};

	// encode storage buffer
	static std::tuple<int, storage::buffer> encode(const ::storage::buffer& buf, unsigned long flags)
	{
		// try all methods
		const size_t num_buffers = 3;

		storage::buffer buffers[num_buffers];

		buffers[storage::fs::ENCODING_NONE] = buf;

		for (size_t i = 1; i < num_buffers; i++)
			buffers[i] = buffers[0];

		if ((flags & FLAG_ENABLE_RLE8) != 0)
			buffers[storage::fs::ENCODING_RLE8] = encoder::rle8::encode(buf);

		if ((flags & FLAG_ENABLE_RLE0) != 0)
			buffers[storage::fs::ENCODING_RLE0] = encoder::rle0::encode(buf);

		// find best
		size_t min_size = buffers[0].size();
		int encoding = 0;

		for (size_t i = 1; i < num_buffers; i++)
		{
			if (buffers[i].size() < min_size)
			{
				min_size = buffers[i].size();
				encoding = (int)i;
			}
		}

		// return
		return std::tuple<int, storage::buffer>(encoding, buffers[encoding]);
	}
	
	// decode storage buffer
	static storage::buffer decode(const ::storage::buffer& buf, int encoding)
	{
		switch (encoding)
		{
		case storage::fs::ENCODING_NONE:
			return buf;

		case storage::fs::ENCODING_RLE8:
			return encoder::rle8::decode(buf);

		case storage::fs::ENCODING_RLE0:
			return encoder::rle0::decode(buf);

		default:
			throw_exception(unknown_encoding_exception, encoding);
		}
	}
};