#pragma once

#include "../utils/singleton.h"
#include "../utils/base64.h"
#include "../storage/buffer.h"
#include "../hash/checksum32.h"
#include "../hash/sha2.h"

#include <atomic>

namespace resources::uid
{
	// create machine-based identification
	storage::buffer ident(void);

	/* manager singleton class */
	class manager : public singleton<manager>
	{
		friend class singleton<manager>;

	public:

		// return uid
		auto uid(void)
		{
			// copy ident
			storage::buffer curr = this->m_ident;

			// add current time for randomness
			{
				auto currtime = std::chrono::time_point_cast<std::chrono::nanoseconds>(std::chrono::high_resolution_clock::now());

				curr.append(currtime.time_since_epoch().count());
			}

			// add thread-safe increment in case clock resolution is not high-enough
			{
				auto currinc = this->m_inc++;

				curr.append(currinc);
			}

			// return in base64
			return utils::base64::encode(sha256(curr));
		}

		// return decorated uid
		auto create(void)
		{
			// format to string
			auto base64 = uid();

			char tmp[128];

			snprintf(tmp, sizeof(tmp), "%%{{%.8X$%s$%.8X}}%%", 'RND0', base64.c_str(), checksum32((const unsigned char*)base64.data(), base64.length()));

			return std::string(tmp);
		}

	private:
		manager(void)
		{
			this->m_inc = 0;

			this->m_ident = ident();
		}

		storage::buffer m_ident;
		std::atomic<uint64_t> m_inc;
	};

	// create UID
	static auto create(void)
	{
		return get_instance<manager>()->create();
	}
};
