#pragma once

#include "../../message/handler.h"

#include "../session/msg.h"
#include "../session/state.h"
#include "../session/base.h"
#include "../session/db.h"

#include "message.h"
#include "client.h"

namespace net::tcp::session
{
	using msg_session_t = net::session::msg_base<net::tcp::msg>;
	using state_t = net::session::state<net::tcp::msg>;
	using base_t = net::session::session_base<net::tcp::msg>;
	using db_base_t = net::session::db_base<net::tcp::msg>;

	/* client class */
	template<const unsigned long IDENT> class client : public net::tcp::client<IDENT>
	{
	public:

		// register message type on ctor
		client(SOCKET socket, db_base_t& db_obj) : net::tcp::client<IDENT>(socket), m_db(db_obj)
		{
			register_messages();
		}

		// register message type on ctor
		client(const char* address, int port, db_base_t& db_obj) : net::tcp::client<IDENT>(address, port), m_db(db_obj)
		{
			register_messages();
		}
		
		~client(void)
		{
			try
			{
				// must call stop here to force synchronize with on_disconnect!
				net::tcp::client<IDENT>::stop();
			}
			catch (...) {}
		}

		size_t num_sessions(void) const
		{
			return this->m_db.count();
		}

	protected:

		virtual void on_disconnect(void)
		{
			// WARNING: offline first, on_disconnect last
			try
			{
				AUTOLOCK(this->m_lock);

				// offline all sessions
				for (auto& v : this->m_ids)
					this->m_db.offline(v);
			}
			catch (...) {}

			net::tcp::client<IDENT>::on_disconnect();
		}

	private:

		// register session message type
		void register_messages(void)
		{
			msg_session_t tmp;

			instancer::local<net::tcp::msg>::add_instancer< net::session::msg_base<net::tcp::msg> >();
			message::handler<net::tcp::msg>::register_listener< net::session::msg_base<net::tcp::msg> >(std::bind(&client<IDENT>::recv_session, this, std::placeholders::_1));
		}

		// add id to watch list if not already into
		void add_id(const std::string& id)
		{
			AUTOLOCK(this->m_lock);

			auto it = std::find(this->m_ids.begin(), this->m_ids.end(), id);

			if (it == this->m_ids.end())
				this->m_ids.push_back(id);
		}

		// handle message type
		void recv_session(const net::session::msg_base<net::tcp::msg>& msg)
		{
			_debug("received message on session %s...", msg.id().c_str());

			// watch this id
			add_id(msg.id());

			// check if session exists
			if (this->m_db.has(msg.id()))
			{
				// process message
				auto p = this->m_db.get(msg.id());

				if (p == nullptr)
					return;

				p->handle(msg.data(), *this);

				return;
			}

			// otherelse create one with idle state
			auto p = this->m_db.create(*this, msg.id());

			p->push_state(p->create_default_state());

			p->handle(msg.data(), *this);
		}

	private:
		mutable std::mutex m_lock;
		std::list<std::string> m_ids;
		db_base_t& m_db;
	};

	/* server class */
	template<const unsigned long IDENT> class server : public net::tcp::server<IDENT>
	{
	protected:
		class client : public net::tcp::session::client<IDENT>, public std::enable_shared_from_this<client>
		{
		public:
			client(server& parent, SOCKET socket, db_base_t& db_obj) : m_server(parent), net::tcp::session::client<IDENT>(socket, db_obj) {}

		private:
			server& m_server;
		};

	public:
		server(const char* address, int port, db_base_t& db_obj) : m_db(db_obj), net::tcp::server<IDENT>(address, port) {}

		size_t num_sessions(void) const
		{
			return this->m_db.count();
		}

	private:
		virtual std::shared_ptr<typename net::tcp::server<IDENT>::client_base_t> create(SOCKET socket) override
		{
			return std::make_shared<client>(*this, socket, this->m_db);
		}

	private:
		db_base_t& m_db;
	};
};

using net_tcp_msg_session_t = net::tcp::session::msg_session_t;
using net_fsm_session_state_tcp = net::tcp::session::state_t;
using net_fsm_session_tcp = net::tcp::session::base_t;

#define REGISTER_TCP_SESSION_MESSAGE_EX(class, type, func)						REGISTER_SESSION_MESSAGE_EX(class, type, func)
#define REGISTER_TCP_SESSION_MESSAGE(class, type)								REGISTER_SESSION_MESSAGE(class, type)

#define HANDLE_TCP_SESSION_MESSAGE_EX(type, obj, endpoint_obj, func)			HANDLE_SESSION_MESSAGE_EX(type, obj, endpoint_obj, net::tcp::msg, func)
#define HANDLE_TCP_SESSION_MESSAGE(type, obj, endpoint)							HANDLE_SESSION_MESSAGE(type, obj, endpoint, net::tcp::msg)

#define REGISTER_TCP_SESSION_MESSAGE_OBJECT(class, type)						REGISTER_SESSION_MESSAGE_OBJECT(class, type)

#define DECLARE_TCP_SESSION_STATE_OBJECT_EX(class, name, type)					DECLARE_SESSION_STATE_OBJECT_EX(class, name, type, net::tcp::msg)
#define DECLARE_TCP_SESSION_STATE_OBJECT(class, type)							DECLARE_SESSION_STATE_OBJECT(class, type, net::tcp::msg)
