#pragma once

#include "../utils/utils.h"
#include "../debug/exception.h"

#include "exception.h"

#include <memory>

#ifdef WIN32

#include <WinSock2.h>
#include <WS2tcpip.h>

#pragma comment(lib, "Ws2_32.lib")

#else

#include <poll.h>

#endif

namespace net
{
	// initialize networking
	static void init(void)
	{
#ifdef WIN32
		WSADATA wsaData;

		if (WSAStartup(MAKEWORD(2, 2), &wsaData))
			throw_exception(cannot_initialize_winsock_exception);
#endif
	}

	// close networking
	static void close(void)
	{
#ifdef WIN32
		WSACleanup();
#endif
	}

	// poll socket
	static bool poll(SOCKET socket, bool* conn_reset, bool* read_ready, bool *write_ready)
	{
		// create poll structure
#ifdef WIN32
		WSAPOLLFD fd;
#else
		struct pollfd fd;
#endif

		// set flags
		fd.fd = socket;
		fd.events = 0;
		fd.revents = 0;

		if (read_ready != nullptr)
			fd.events |= POLLIN;

		if (write_ready != nullptr)
			fd.events |= POLLOUT;

		// poll socket
#ifdef WIN32
		if (WSAPoll(&fd, 1, 0) != 1)
			return false;
#else
		if (poll(&fd, 1, 0) != 1)
			return false;
#endif

		// check flags
		if (conn_reset != nullptr)
			*conn_reset = (fd.revents & (POLLERR | POLLHUP | POLLNVAL)) != 0;

		if (read_ready != nullptr)
			*read_ready = (fd.revents & POLLIN) != 0;

		if(write_ready != nullptr)
			*write_ready = (fd.revents & POLLOUT) != 0;

		// return successfuly
		return true;
	}
};