#pragma once

#include "../solve/base.h"
#include "../solve/var.h"
#include "../solve/math.h"

#include "../../resources/uid.h"

#include "../../exception.h"

namespace math::symbolic
{
	/* symbolic variable class wraps a solve pointer */
	template<typename type> class var
	{
	public:
		// default ctor
		var(void) {}

		// ctor by value
		var(const type value)
		{
			this->m_pointer = std::make_shared< math::solve::fixed<type> >(value);
		}

		// ctor by pointer
		var(std::shared_ptr< const math::solve::base<type> > pointer)
		{
			this->m_pointer = pointer;
		}

		// ctor by object
		var(const var<type>& obj)
		{
			this->m_pointer = obj.m_pointer;
		}

		// get value
		const type eval(void) const
		{
			// transform null to zero
			if (this->m_pointer == nullptr)
				return (type)0;

			return this->m_pointer->eval();
		}

		// get derivative
		var<type> get_derivative(const math::solve::base<type> *p) const
		{
			// 0' = 0
			if (this->m_pointer == nullptr)
				return var(nullptr);

			return var(this->m_pointer->get_derivative(p));
		}

		// get derivative
		var<type> get_derivative(std::shared_ptr< const math::solve::base<type> > p) const
		{
			return get_derivative(p.get());
		}

		// get derivative
		var<type> get_derivative(const var<type>& obj) const
		{
			return get_derivative(obj.ptr());
		}

		// retrieve const pointer
		std::shared_ptr< const math::solve::base<type> > ptr(void) const
		{
			return this->m_pointer;
		}

	private:
		std::shared_ptr< const math::solve::base<type> > m_pointer;
	};

	template<typename type> var<type> operator+(const var<type>& a, const var<type>& b)
	{
		return var<type>(std::make_shared< math::solve::add<type> >(a.ptr(), b.ptr()));
	}

	template<typename type> var<type> operator+(const var<type>& a, const type b)
	{
		return var<type>(std::make_shared< math::solve::add_const<type> >(a.ptr(), b));
	}

	template<typename type> var<type> operator+(const type a, const var<type>& b)
	{
		return var<type>(std::make_shared< math::solve::const_add<type> >(a, b.ptr()));
	}

	template<typename type> var<type> operator-(const var<type>& a, const var<type>& b)
	{
		return var<type>(std::make_shared< math::solve::subtract<type> >(a.ptr(), b.ptr()));
	}

	template<typename type> var<type> operator-(const var<type>& a, const type b)
	{
		return var<type>(std::make_shared< math::solve::subtract_const<type> >(a.ptr(), b));
	}

	template<typename type> var<type> operator-(const type a, const var<type>& b)
	{
		return var<type>(std::make_shared< math::solve::const_subtract<type> >(a, b.ptr()));
	}

	template<typename type> var<type> operator*(const var<type>& a, const var<type>& b)
	{
		return var<type>(std::make_shared< math::solve::mult<type> >(a.ptr(), b.ptr()));
	}

	template<typename type> var<type> operator*(const var<type>& a, const type b)
	{
		return var<type>(std::make_shared< math::solve::mult_const<type> >(a.ptr(), b));
	}

	template<typename type> var<type> operator*(const type a, const var<type>& b)
	{
		return var<type>(std::make_shared< math::solve::const_mult<type> >(a, b.ptr()));
	}

	template<typename type> var<type> operator/(const var<type>& a, const var<type>& b)
	{
		return var<type>(std::make_shared< math::solve::div<type> >(a.ptr(), b.ptr()));
	}

	template<typename type> var<type> operator/(const var<type>& a, const type b)
	{
		return var<type>(std::make_shared< math::solve::div_const<type> >(a.ptr(), b));
	}

	template<typename type> var<type> operator/(const type a, const var<type>& b)
	{
		return var<type>(std::make_shared< math::solve::const_div<type> >(a, b.ptr()));
	}

	template<typename type> var<type> operator-(const var<type>& obj)
	{
		return var<type>(std::make_shared< math::solve::negate<type> >(obj.ptr()));
	}

	template<typename type> var<type> sqrt(const var<type>& obj)
	{
		return var<type>(std::make_shared< math::solve::sqrt<type> >(obj.ptr()));
	}

	template<typename type> var<type> square(const var<type>& obj)
	{
		return var<type>(std::make_shared< math::solve::square<type> >(obj.ptr()));
	}
};