#pragma once

#include "../../debug/exdebug.h"
#include "../../utils/singleton.h"
#include "../../event/auto_update.h"

#include "../../exception.h"

#include "../solve/base.h"

#include <unordered_map>
#include <map>
#include <memory>
#include <string>

namespace math::symbolic
{
	enum class sensitivity
	{
		none,
		absolute,
		proportional,
	};

	/* solve::manager class */
	template<typename type> class collection
	{
		struct data_s
		{
			std::shared_ptr<const math::solve::base<type>> pointer;
			type dvalue;
			sensitivity dtype;
		};

	public:

		// default ctor
		collection(const std::string& name)
		{
			_debug("creating variable collection %s", name.c_str());

			this->m_name = name;
		}

		// retrieve name
		const std::string& name(void) const
		{
			return this->m_name;
		}

		// reset
		void reset(void)
		{
			_debug("reseting variable collection %s", this->m_name.c_str());

			this->m_list.clear();
		}

		// check if variable exists
		bool has(const std::string& name) const
		{
			// check current list
			if (this->m_list.find(name) != this->m_list.end())
				return true;

			// check children
			for (auto& v : this->m_children)
				if (v.second != nullptr && v.second->has(name))
					return true;

			return false;
		}

		// check if subgroup exists
		bool has_subgroup(const std::string& name) const
		{
			return this->m_children.find(name) != this->m_children.end();
		}

		// create subgroup
		collection<type>& create_subgroup(const std::string& name)
		{
			_debug("creating subgroup %s in collection %s", name.c_str(), this->m_name.c_str());

			// throw exception if subgroup already exists
			if(has_subgroup(name))
				throw_exception(name_already_exist_exception, name);

			// add to chilren
			this->m_children.emplace(std::make_pair(name, std::make_shared<collection<type>>(name)));

			return get_subgroup(name);
		}

		// get subgroup
		collection<type>& get_subgroup(const std::string& name)
		{
			// retrieve local
			auto it = this->m_children.find(name);

			if (it != this->m_children.end())
				return *(it->second);

			// throw error if not found
			throw_exception(name_not_found_exception, name);
		}

		// get subgroup
		const collection<type>& get_subgroup(const std::string& name) const
		{
			// retrieve local
			auto it = this->m_children.find(name);

			if (it != this->m_children.end())
			{
				if (it->second == nullptr)
					throw_exception(null_pointer_exception);

				return *it->second;
			}

			// throw error if not found
			throw_exception(name_not_found_exception, name);
		}

		// add var
		std::shared_ptr<const math::solve::base<type>> add(const std::string& name, const var<type>& obj, const type dvalue = 0, math::symbolic::sensitivity dtype = math::symbolic::sensitivity::none)
		{
			return add(name, obj.ptr(), dvalue, dtype);
		}

		// add solve element
		std::shared_ptr<const math::solve::base<type>> add(const std::string& name, std::shared_ptr<const math::solve::base<type>> p, const type dvalue = 0, math::symbolic::sensitivity dtype = math::symbolic::sensitivity::none)
		{
			_debug("adding variable %s to collection %s", name.c_str(), this->m_name.c_str());

			// discard null pointers
			if (__ISNULLPTR(p))
				return nullptr;

			// check if name already exists
			if (has(name))
				throw_exception(name_already_exist_exception, name);

			// add to list
			struct data_s s;

			s.pointer = p;
			s.dvalue = dvalue;
			s.dtype = dtype;

			this->m_list.emplace(std::make_pair(name, s));

			// return pointer
			return p;
		}

		// list subgroups
		std::list<std::string> list_subgroups(void) const
		{
			std::list<std::string> ret;

			for (auto& v : this->m_children)
				ret.push_back(v.first);

			return ret;
		}

		// return list of all cumulated vars
		std::list<std::string> list_all_names(void) const
		{
			std::list<std::string> ret;

			dump_names(ret, true);

			return ret;
		}

		// return list of all local vars
		std::list<std::string> list_local_names(void) const
		{
			std::list<std::string> ret;

			dump_names(ret, false);

			return ret;
		}

		// get sensitivity
		type get_deviation(const var<type>& var, const std::string& name) const
		{
			// get variable
			auto obj = get(name);

			// compute sensitivity
			if (obj.pointer == nullptr)
				return (type)0;

			type dvalue = 0;

			switch (obj.dtype)
			{
			case sensitivity::absolute:
				dvalue = obj.dvalue;
				break;

			case sensitivity::proportional:
				dvalue = obj.dvalue * obj.pointer->eval();
				break;

			case sensitivity::none:
				break;
			}

			// return 0 if no sensitivity
			if (dvalue == 0)
				return (type)0;

			// evaluate contribution from derivative and sensitivity f'df
			auto dval = var.get_derivative(obj.pointer.get());

			return dval.eval() * dvalue;
		}

	private:

		// get element
		auto get(const std::string& name) const
		{
			// retrieve local
			auto it = this->m_list.find(name);

			if (it != this->m_list.end())
				return it->second;

			// retrieve child
			for (auto& v : this->m_children)
			{
				if (v.second == nullptr)
					continue;

				if (v.second->has(name))
					return v.second->get(name);
			}

			// throw error if not found in any
			throw_exception(name_not_found_exception, name);
		}

		// dump all names
		void dump_names(std::list<std::string>& list, bool recursive = true) const
		{
			// dump children
			if (recursive)
			{
				for (auto& v : this->m_children)
					if (v.second != nullptr)
						v.second->dump_names(list, recursive);
			}

			// dump current
			for (auto& v : this->m_list)
				list.push_back(v.first);
		}

	private:
		std::string m_name;

		std::unordered_map<std::string, struct data_s> m_list;
		std::unordered_map<std::string, std::shared_ptr<collection<type>>> m_children;
	};
};