#pragma once

#include "../../symbolic/var.h"

#include "yu.h"
#include "wavefront.h"

namespace math::optics::paraxial
{
	/* seidel aberration class */
	template<typename type> class seidel
	{
	public:

		// default ctor has null values
		seidel(void) {}

		// ctor from variables
		seidel(const math::symbolic::var<type>& _spherical, const math::symbolic::var<type>& _coma, const math::symbolic::var<type>& _astig, const math::symbolic::var<type>& _petzval, const math::symbolic::var<type>& _dist)
		{
			this->m_data[0] = _spherical;
			this->m_data[1] = _coma;
			this->m_data[2] = _astig;
			this->m_data[3] = _petzval;
			this->m_data[4] = _dist;
		}

		// create from ray
		seidel(const yu<type>& yu_marginal_before, const yu<type>& yu_marginal_after, const yu<type>& yu_chief_before, const yu<type>& yu_chief_after, const math::symbolic::var<type>& curv, const math::symbolic::var<type>& n_before, const math::symbolic::var<type>& n_after)
		{
			// short-hand notations
			const auto& h = yu_chief_before.y();
			const auto& u = yu_chief_before.u();

			const auto& hp = yu_marginal_before.y();
			const auto& up = yu_marginal_before.u();

			// consts
			auto A = n_before * (h * curv + u);
			auto Ap = n_before * (hp * curv + up);

			auto dun = (yu_chief_after.u() / n_after - yu_chief_before.u() / n_before);
			auto dn = ((type)1 / n_after - (type)1 / n_before);

			auto L = n_before * (h * up - hp * u);
			auto L2 = math::symbolic::square(L);

			auto fact = Ap / A;

			// compute Seidel terms as compactly as possible
			this->m_data[0] = -h * A * A * dun;
			this->m_data[1] = this->m_data[0] * fact;
			this->m_data[2] = this->m_data[1] * fact;
			this->m_data[3] = -curv * L2 * dn;
			this->m_data[4] = (this->m_data[2] + this->m_data[3]) * fact;
		}

		// get spherical aberration
		const math::symbolic::var<type>& spherical(void) const
		{
			return this->m_data[0];
		}

		// get coma
		const math::symbolic::var<type>& coma(void) const
		{
			return this->m_data[1];
		}

		// get astigmatism
		const math::symbolic::var<type>& astig(void) const
		{
			return this->m_data[2];
		}

		// get petzval
		const math::symbolic::var<type>& petzval(void) const
		{
			return this->m_data[3];
		}

		// get distortion
		const math::symbolic::var<type>& dist(void) const
		{
			return this->m_data[4];
		}

		// craete wavefront from seidel coefficients
		wavefront<type> to_wavefront(void) const
		{
			return wavefront<type>(
				this->m_data[0] / (type)8,
				this->m_data[1] / (type)2,
				this->m_data[2] / (type)2,
				(this->m_data[2] + this->m_data[3]) / (type)4,
				this->m_data[4] / (type)2
			);
		}

	private:
		math::symbolic::var<type> m_data[5];
	};

	// sum two seidel
	template<typename type> seidel<type> operator+(const seidel<type>& a, const seidel<type>& b)
	{
		return seidel<type>(
			a.spherical() + b.spherical(),
			a.coma() + b.coma(),
			a.astig() + b.astig(),
			a.petzval() + b.petzval(),
			a.dist() + b.dist()
		);
	}
};