#pragma once

#include "../utils/utils.h"
#include "../debug/exception.h"
#include "../debug/evemon.h"
#include "../debug/exdebug.h"

#include "exception.h"
#include "base.h"

#include <string>

#ifdef _WIN32
#include <Windows.h>
#else
#include <dlfcn.h>
#endif

namespace library
{
	/* shared object libraries */
	class so : public base
	{
	public:

		// acquire handle on ctor
		so(const std::string& filename)
		{
			_debug("loading shared object \"%s\"...", filename.c_str());

#ifdef _WIN32
			this->m_handle = LoadLibraryA(filename.c_str());

			if (this->m_handle == NULL)
				throw_exception(invalid_library_exception, filename);
#else
			this->m_handle = dlopen(filename.c_str(), RTLD_LAZY);

			if (this->m_handle == nullptr)
				throw_exception(invalid_library_exception, filename);
#endif
		}

		// free handle on dtor
		~so(void)
		{
#ifdef _WIN32
			if (this->m_handle != NULL)
				FreeLibrary(this->m_handle);

			this->m_handle = NULL;
#else
			if (this->m_handle != nullptr)
				dlclose(this->m_handle);

			this->m_handle = nullptr;
#endif
		}

		// get address
		virtual void* get_addr(const std::string& funcname) const override
		{
#ifdef _WIN32
			auto p = GetProcAddress(this->m_handle, funcname.c_str());
#else
			auto p = dlsym(this->m_handle, funcname.c_str());
#endif

			if (__ISNULLPTR(p))
				throw_exception(unknown_func_exception, funcname);

			return p;
		}

	private:

#ifdef _WIN32
		HMODULE m_handle;
#else
		void* m_handle;
#endif
	};
};
