#pragma once

#include "../../utils/utils.h"
#include "../../utils/singleton.h"
#include "../../debug/exdebug.h"
#include "../../debug/evemon.h"

#include "exception.h"
#include "base.h"

#include <string>
#include <memory>
#include <unordered_map>

namespace library::interface
{
	/* library interface manager singleton class */
	class manager : public singleton<manager>
	{
		friend class singleton<manager>;

	public:

		// check if interface exists
		template<class type> bool has(void) const
		{
			auto type_name = std::string(typeid(type).name());

			// check if existing
			return this->m_interfaces.find(type_name) != this->m_interfaces.end();
		}

		// retrieve current interface for type
		template<class type> std::shared_ptr<const type> get(void) const
		{
			auto type_name = std::string(typeid(type).name());

			// check if existing
			auto it = this->m_interfaces.find(type_name);

			if (it == this->m_interfaces.end())
				throw_exception(unknown_interface_exception, type_name);

			// get casted pointer
			auto p = std::static_pointer_cast<const type>(it->second);

			// throw exception on invalid interface
			if (__ISNULLPTR(p))
				throw_exception(null_interface_exception);

			return p;
		}

		// change current interface for type
		template<class type> void set(std::shared_ptr<const type> p)
		{
			// throw exception on invalid interface
			if (__ISNULLPTR(p))
				throw_exception(null_interface_exception);

			// get type name
			auto type_name = std::string(typeid(type).name());

			// check if already existing
			auto it = this->m_interfaces.find(type_name);

			if (it != this->m_interfaces.end())
			{
				it->second = p;

				return;
			}

			// insert into list
			_debug("registering interface \"%s\"", type_name.c_str());

			this->m_interfaces.emplace(std::make_pair(type_name, p));
		}

		// detach all interfaces
		void detach_all(void)
		{
			this->m_interfaces.clear();
		}

		// auto_register interface
		template<class type> class auto_register
		{
		public:
			auto_register(std::shared_ptr<const type> p)
			{
				::get_instance<library::interface::manager>()->set<type>(p);
			}
		};

	private:
		std::unordered_map<std::string, std::shared_ptr<const library::interface::base>> m_interfaces;
	};
};