#pragma once

#include "../utils/utils.h"
#include "../debug/evemon.h"
#include "../debug/exdebug.h"
#include "../utils/singleton.h"
#include "../debug/safe.h"
#include "../threads/lock.h"
#include "../threads/thread.h"

#include "jobs.h"

#include <list>

#define DEFAULT_DEFAULT_NUM_WORKER_THREADS	0
#define DEFAULT_MAX_INACTIVITY_TIME			10.0

/* thread_pool_manager class */
class thread_pool_manager : public singleton<thread_pool_manager>
{
public:
	friend class singleton<thread_pool_manager>;

private:
	// worker thread
	class worker_thread : public thread::base, public std::enable_shared_from_this<worker_thread>
	{
	public:
		worker_thread(double max_inactivity_time=0)
		{
			_debug("creating worker thread %p", this);

			this->m_max_inactivity_time = max_inactivity_time;

			this->m_stop = false;
		}

		virtual ~worker_thread(void)
		{
			_debug("deleting worker thread %p", this);

			try
			{
				clean_stop();
			}
			catch (...) {}
		}

		// restart
		void restart(void)
		{
			_debug("restarting worker thread %p", this);

			this->m_stop = false;

			stop();
			start();
		}

		// clean stop
		void clean_stop(void)
		{
			this->m_stop = true;
			this->m_work_pending.trigger();

			stop();
		}

		// signal stop
		virtual bool stop_condition(void) const override
		{
			return this->m_stop;
		}

		// check if no job is currently assigned
		bool is_free(void) const
		{
			return std::atomic_load(&this->m_job) == nullptr;
		}

		// assign a job
		void assign(std::shared_ptr<job_base> p)
		{
			std::atomic_store(&this->m_job, p);

			// trigger wait event
			this->m_work_pending.trigger();
		}

	private:

		// run job
		virtual void run(void) override;

		volatile std::atomic<bool> m_stop;

		std::shared_ptr<job_base> m_job;
		std::atomic<double> m_max_inactivity_time;

		thread::event m_work_pending;
	};

public:

	// set number of maximum threads
	void set_max_workers(size_t max_worker_threads)
	{
		_debug("changing maximum number of worker thread to %zu", max_worker_threads);

		this->m_max_worker_threads = max_worker_threads;
	}

	// set maximum inactivity time
	void set_max_inactivity_time(double max_inactivity_time)
	{
		_debug("changing maximum inactivity time for worker thread to %.1e seconds", max_inactivity_time);

		this->m_max_inactivity_time = max_inactivity_time;
	}

	// queue a job
	void queue(std::shared_ptr<job_base> p)
	{
		if (__ISNULLPTR(p))
			return;

		AUTOLOCK(this->m_lock);

		// run job if no threads allowed
		if (this->m_max_worker_threads == 0)
		{
			p->run();

			return;
		}

		// until a solution is found
		do
		{
			auto _browse_workers = [](const std::list<std::shared_ptr<worker_thread>>& list, std::shared_ptr<job_base> job)
			{
				for (auto& p : list)
				{
					// skip null
					if (p == nullptr)
						continue;

					// skip busy
					if (!p->is_free())
						continue;

					_debug("assigning job to worker thread %p", p.get());

					// assign job and exit
					p->assign(job);

					// restart if stopped
					if (!p->is_running())
						p->restart();

					return true;
				}

				return false;
			};

			// browse threads list
			if (_browse_workers(this->m_worker_threads, p) || _browse_workers(this->m_extra_threads, p))
				return;

			// otherelse, and if below max number of threads, create a new one
			if (this->m_worker_threads.size() < this->m_max_worker_threads)
			{
				auto wthread = std::make_shared<worker_thread>(this->m_max_inactivity_time);

				if (wthread == nullptr)
					continue;

				this->m_worker_threads.push_back(wthread);

				_debug("creating new worker thread %p", wthread.get());

				wthread->assign(p);
				wthread->start();

				// exit
				return;
			}

			// last resort is to wait and retry
			std::this_thread::sleep_for(std::chrono::milliseconds(10));

		} while (true);
	}

	// compute capacity
	int get_capacity(void) const
	{
		size_t cnt = 0;

		{
			AUTOLOCK(this->m_lock);

			// count number of active thread
			for (auto& p : this->m_worker_threads)
				if (p != nullptr && p->is_running() && !p->is_free())
					cnt++;
		}

		// should not occur
		if (cnt > this->m_max_worker_threads)
			return -MAKE_INT(cnt - this->m_max_worker_threads);

		// return difference between max allowed number of threads and count
		return MAKE_INT(this->m_max_worker_threads - cnt);
	}

	// reset all
	void reset(void)
	{
		// stop all threads
		for (auto& p : this->m_worker_threads)
			if (!__ISNULLPTR(p))
				p->clean_stop();

		for (auto& p : this->m_extra_threads)
			if (!__ISNULLPTR(p))
				p->clean_stop();

		// clear list
		this->m_worker_threads.clear();
		this->m_extra_threads.clear();
	}

	// temporarily allow one more thread
	auto allow_more(void)
	{
		class extra_thread_allowance
		{
		public:
			extra_thread_allowance(std::shared_ptr<worker_thread> p)
			{
				this->m_thread = p;
			}

			extra_thread_allowance(extra_thread_allowance&& other) noexcept
			{
				this->m_thread = std::move(other.m_thread);
			}

			extra_thread_allowance(const extra_thread_allowance&) = delete;

			const extra_thread_allowance& operator=(const extra_thread_allowance&) = delete;

			~extra_thread_allowance(void)
			{
				if (this->m_thread != nullptr)
					this->m_thread->clean_stop();

				this->m_thread = nullptr;
			}

		private:
			std::shared_ptr<worker_thread> m_thread;
		};

		// create thread with infinite timing
		auto p = std::make_shared<worker_thread>(0);

		_debug("creating a new temporary thread %p", p.get());

		p->start();

		// add to list
		{
			AUTOLOCK(this->m_lock);

			this->m_extra_threads.push_back(p);
		}

		// return object that will self kill thread once destructed
		return extra_thread_allowance(p);
	}

private:

	// initialize thread pool manager
	thread_pool_manager(void)
	{
		this->m_max_worker_threads = DEFAULT_DEFAULT_NUM_WORKER_THREADS;
		this->m_max_inactivity_time = DEFAULT_MAX_INACTIVITY_TIME;
	}

	// remove thread from list
	void remove(std::shared_ptr<worker_thread> p)
	{
		_debug("removing worker thread %p", p.get());

		if (__ISNULLPTR(p))
			return;

		{
			AUTOLOCK(this->m_lock);

			this->m_worker_threads.remove(p);
		}

		p->clean_stop();
	}

	// list of worker threads
	std::atomic<size_t> m_max_worker_threads;
	std::atomic<double> m_max_inactivity_time;

	std::list< std::shared_ptr<worker_thread> > m_worker_threads;
	std::list< std::shared_ptr<worker_thread> > m_extra_threads;

	// single producer only
	mutable lock_t m_lock;
};
