#pragma once

#include "../utils/utils.h"
#include "../debug/evemon.h"
#include "../types/type.h"

#include "exception.h"

#include <list>
#include <string>
#include <memory>

namespace instancer
{
	/* instancer::local class */
	template<class type_base, typename ... params> class local
	{
	private:

		/* instancer_base */
		class instancer_base
		{
		public:
			virtual bool is_type(const std::string& type_name) const = 0;
			virtual std::shared_ptr<type_base> create_instance(params ... args) const = 0;
		};

		/* instancer class */
		template<class type> class instancer : public instancer_base
		{
		public:
			// return true if same type
			virtual bool is_type(const std::string& type_name) const override
			{
				return ::is_type_name<type>(type_name);
			}

			virtual std::shared_ptr<type_base> create_instance(params ... args) const override
			{
				return std::make_shared<type>(args ...);
			}
		};

	public:

		// create instance based on type
		std::shared_ptr<type_base> create_instance(const std::string& type_name, params ... args)
		{
			for (auto& p : this->m_list)
			{
				if (__ISNULLPTR(p))
					continue;

				if (p->is_type(type_name))
					return p->create_instance(args ...);
			}

			throw_exception(unknown_instancer_exception, type_name);
		}

	protected:

		// add instancer based on type
		template<typename type> void add_instancer(void)
		{
			this->m_list.emplace_back(std::make_unique<instancer<type>>());
		}

	private:

		// list of known instancers
		std::list< std::unique_ptr<instancer_base> > m_list;
	};
};

#define DECLARE_LOCAL_INSTANCER(type)		this->add_instancer<type>()