#pragma once

#include "../utils/utils.h"
#include "../debug/evemon.h"
#include "../debug/exdebug.h"
#include "../utils/singleton.h"
#include "../utils/singleton_manager.h"
#include "../library/library.h"
#include "../types/type.h"

#include "exception.h"

#include <memory>
#include <string>
#include <unordered_map>
#include <list>
#include <functional>

namespace instancer
{
	/* instanciable object interface class */
	template<class type, typename ... params> class object_base
	{
	public:
		template<class sub_class> static std::shared_ptr<type> create(params ... args)
		{
			return std::make_shared<sub_class>(args ...);
		}
	};

	/* instancer class given type */
	template<typename type_ret, typename ... params> class global : public singleton< global<type_ret, params...> >
	{
		friend class singleton<global<type_ret, params...>>;

	public:
		using instancer_t = std::function<std::shared_ptr<type_ret>(params ...)>;

		/* auto register class for given type */
		class auto_register
		{
		public:
			auto_register(const std::string& name, instancer_t callback)
			{
				auto p = instancer::global<type_ret, params ...> ::get_instance();

				if (__ISNULLPTR(p))
					return;

				p->add(name, callback);
			}
		};

		// create instance by name
		std::shared_ptr<type_ret> create_instance(const std::string& name, params ... args) const
		{
			// get callback function
			auto it = this->m_instancers.find(name);

			// throw exception if instancer is not found
			if (it == this->m_instancers.end())
				throw_exception(unknown_instancer_exception, name);

			// throw error if callback is invalid
			if (!it->second)
				throw_exception(unknown_instancer_exception, name);

			// otherelse call instancer function
			return it->second(args ...);
		}

		// return true if exists
		bool has(const std::string& name) const
		{
			return this->m_instancers.find(name) != this->m_instancers.end();
		}

		// list all registered entries
		std::list<std::string> list(void) const
		{
			std::list<std::string> ret;

			for (auto& v : this->m_instancers)
				ret.push_back(v.first);

			return ret;
		}

	private:

		// add instancer to list
		void add(const std::string& name, instancer_t callback)
		{
			// throw error if callback is invalid
			if (!callback)
				throw_exception(invalid_instancer_callback_exception, name);

			// push to list
			this->m_instancers.insert(std::make_pair(name, callback));
		}

		std::unordered_map<std::string, instancer_t> m_instancers;
	};
};

// create instance
template<class type, typename ... params> std::shared_ptr<type> create_instance(const std::string& name, params ... args)
{
	auto s = split_typename(name);

	if(s.lib == "")
		return get_instance<instancer::global<type, params...>>()->create_instance(s.type, args ...);

	return get_remote_singleton<instancer::global<type, params ...>>(s.lib)->create_instance(s.type, args ...);
}

// declare instancer for given type
#define DECLARE_INSTANCER(type)											INITIALIZE_SINGLETON_EX(instancer::global<type>, instancer__##type);						\
																		BIND_SINGLETON_EX(instancer::global<type>, instancer_global_##type);

// declare instanceable object for given class/type
#define DECLARE_INSTANCEABLE_OBJECT_EX(class, name, type)				std::shared_ptr<type> create_instance__##name(void)											\
																		{																							\
																			return type :: create<class>();															\
																		}																							\
																																									\
																		REGISTER_TYPENAME_EX(class, #class, instancer_type_##name);									\
																																									\
																		instancer::global< type > :: auto_register g_register_instancer__##name(::get_name_from_type< class >(), create_instance__##name);

#define DECLARE_INSTANCEABLE_OBJECT(class, type)						DECLARE_INSTANCEABLE_OBJECT_EX(class, type##class, type)