#pragma once

#include "../debug/safe.h"

#include "map2d.h"

enum class connexity
{
	CONN_4,
	CONN_8,
};

static map2d<bool> imerode(const map2d<bool>& map, const map2d<bool>& str_el)
{
	map2d<bool> ret(map.width(), map.height());

	size_t str_el_width2 = str_el.width() >> 1;
	size_t str_el_height2 = str_el.height() >> 1;

	ret.perpixel([&](size_t x, size_t y)
		{
			if (!map(x, y))
				return false;

			// scan all pixels around structuring element
			for (size_t ofs_y = 0; ofs_y < str_el.height(); ofs_y++)
				for (size_t ofs_x = 0; ofs_x < str_el.width(); ofs_x++)
				{
					// skip if pixel is not in structuring element
					if (!str_el(ofs_x, ofs_y))
						continue;

					// skip if x or y is lower than offset center
					if (MAKE_ADD(x, ofs_x) < str_el_width2 || MAKE_ADD(y, ofs_y) < str_el_height2)
						continue;

					// new coordinates
					size_t x2 = MAKE_SUB(MAKE_ADD(x, ofs_x), str_el_width2);
					size_t y2 = MAKE_SUB(MAKE_ADD(y, ofs_y), str_el_height2);

					// skip if beyond dimensions
					if (x2 >= map.width() || y2 >= map.height())
						continue;

					// erode if any pixel is in mask and not in map
					if (!map(x2, y2))
						return false;
				}

			return true;
		});

	return ret;
}

static map2d<bool> imdilate(const map2d<bool>& map, const map2d<bool>& str_el)
{
	map2d<bool> ret(map.width(), map.height());

	size_t str_el_width2 = str_el.width() >> 1;
	size_t str_el_height2 = str_el.height() >> 1;

	ret.perpixel([&](size_t x, size_t y)
		{
			if (map(x, y))
				return true;

			// scan all pixels around structuring element
			for (size_t ofs_y = 0; ofs_y < str_el.height(); ofs_y++)
				for (size_t ofs_x = 0; ofs_x < str_el.width(); ofs_x++)
				{
					// skip if pixel is not in structuring element
					if (!str_el(ofs_x, ofs_y))
						continue;

					// skip if x or y is lower than offset center
					if (MAKE_ADD(x, ofs_x) < str_el_width2 || MAKE_ADD(y, ofs_y) < str_el_height2)
						continue;

					// new coordinates
					size_t x2 = MAKE_SUB(MAKE_ADD(x, ofs_x), str_el_width2);
					size_t y2 = MAKE_SUB(MAKE_ADD(y, ofs_y), str_el_height2);

					// skip if beyond dimensions
					if (x2 >= map.width() || y2 >= map.height())
						continue;

					// erode if any pixel is in mask and not in map
					if (map(x2, y2))
						return true;
				}

			return false;
		});

	return ret;
}

static map2d<bool> imopen(const map2d<bool>& map, const map2d<bool>& str_el)
{
	return imdilate(imerode(map, str_el), str_el);
}

static map2d<bool> imclose(const map2d<bool>& map, const map2d<bool>& str_el)
{
	return imerode(imdilate(map, str_el), str_el);
}

static map2d<bool> bwperim(const map2d<bool>& map, const connexity econnexity=connexity::CONN_4)
{
	auto _get = [&](long x, long y)
	{
		if (x < 0 || y < 0 || x >= MAKE_LONG(map.width()) || y >= MAKE_LONG(map.height()))
			return true;

		return map(x, y);
	};

	map2d<bool> ret(map.width(), map.height());

	ret.perpixel([&](size_t x, size_t y)
		{
			// skip if pixel is already outside
			if (!map(x, y))
				return false;

			// check for any zeros around current pixel
			// scan all pixels around structuring element
			bool is_inside = true;

			int lx = MAKE_LONG(x);
			int ly = MAKE_LONG(y);

			switch (econnexity)
			{
				case connexity::CONN_4:
					is_inside &= _get(MAKE_SUB(lx, 1), ly);
					is_inside &= _get(MAKE_ADD(lx, 1), ly);
					is_inside &= _get(lx, MAKE_SUB(ly, 1));
					is_inside &= _get(lx, MAKE_ADD(ly, 1));
					break;

				case connexity::CONN_8:
					is_inside &= _get(MAKE_SUB(lx, 1), MAKE_ADD(ly, 1));
					is_inside &= _get(MAKE_SUB(lx, 1), MAKE_SUB(ly, 1));
					is_inside &= _get(MAKE_ADD(lx, 1), MAKE_ADD(ly, 1));
					is_inside &= _get(MAKE_ADD(lx, 1), MAKE_SUB(ly, 1));

					is_inside &= _get(MAKE_SUB(lx, 1), ly);
					is_inside &= _get(MAKE_ADD(lx, 1), ly);
					is_inside &= _get(lx, MAKE_SUB(ly, 1));
					is_inside &= _get(lx, MAKE_ADD(ly, 1));
					break;

				default:
					is_inside = false;
			}

			return is_inside;
		});

	return ret;
}

/*
static map2d<bool> imfill(const map2d<bool>& map)
{
	map2d<bool> ret(map.width(), map.height());

	for (size_t y = 0; y < ret.height(); y++)
		for (size_t x = 0; x < ret.width(); x++)
		{
		}

	return ret;
}
*/