#pragma once

#include "../debug/evemon.h"

#include "../exception.h"

#include "map2d.h"

#include <cmath>

#ifdef WIN32
#pragma pack(push)
#pragma pack(1)
#endif

struct bitmap_file_header_s
{
	uint16_t bfType;
	uint32_t bfSize;
	uint16_t bfReserved1;
	uint16_t bfReserved2;
	uint32_t bfOffBits;

#ifdef WIN32
};
#pragma pack(pop)
#else
} __attribute__((__packed__));
#endif

#ifdef WIN32
#pragma pack(push)
#pragma pack(1)
#endif

struct bitmap_info_header_s
{
	uint32_t biSize;
	int32_t biWidth;
	int32_t biHeight;
	uint16_t biPlanes;
	uint16_t biBitCount;
	uint32_t biCompression;
	uint32_t biSizeImage;
	int32_t biXPelsPerMeter;
	int32_t biYPelsPerMeter;
	uint32_t biClrUsed;
	uint32_t biClrImportant;
#ifdef WIN32
};
#pragma pack(pop)
#else
} __attribute__((__packed__));
#endif

static void save_bitmap(const map2d<unsigned char>& data, const std::string& filename, const double dpi=72)
{
	struct bitmap_file_header_s bmp_header;
	struct bitmap_info_header_s bmp_info;

	// debug
	_debug("Writing bitmap \"%s\"", filename.c_str());

	// open file stream
	std::ofstream file(filename, std::ios::out | std::ios::binary | std::ios::trunc);

	// exit if cannot open file
	if (!file.is_open())
		throw_exception(cannot_open_file_exception, filename);

	// number of elements
	size_t strides = data.width();

	if (strides % 4 != 0)
		strides += 4 - (strides % 4);

	// create header
	bmp_header.bfOffBits = sizeof(struct bitmap_file_header_s) + sizeof(struct bitmap_info_header_s) + 256 * 4;
	bmp_header.bfReserved1 = 0;
	bmp_header.bfReserved2 = 0;
	bmp_header.bfType = 'MB';
	bmp_header.bfSize = (uint32_t)(sizeof(struct bitmap_file_header_s) + sizeof(struct bitmap_info_header_s) + (strides * data.height()) + 256 * 4);

	file.write((char*)&bmp_header, sizeof(bmp_header));

	// create info struct
	bmp_info.biBitCount = 8;
	bmp_info.biClrImportant = 0;
	bmp_info.biClrUsed = 0;
	bmp_info.biCompression = /*BI_RGB*/0x0000;
	bmp_info.biHeight = (int32_t)data.height();
	bmp_info.biPlanes = 1;
	bmp_info.biSize = sizeof(struct bitmap_info_header_s);
	bmp_info.biSizeImage = 0;
	bmp_info.biWidth = (int32_t)data.width();
	bmp_info.biXPelsPerMeter = (int32_t)std::round(1000.0 * dpi / 25.4);
	bmp_info.biYPelsPerMeter = bmp_info.biXPelsPerMeter;

	file.write((char*)&bmp_info, sizeof(bmp_info));

	// write palette
	for (int i = 0; i < 256; i++)
	{
		file.put((char)i);
		file.put((char)i);
		file.put((char)i);
		file.put((char)0xff);
	}

	// write all data
	for (size_t y = 0; y < data.height(); y++)
	{
		// write row
		for (size_t x = 0; x < data.width(); x++)
			file.put(data(x,y));

		// add padding
		for (size_t x = data.width(); x < strides; x++)
			file.put((char)0);
	}
}
